/*
	Copyright (C) 2012 Steve Thomas <steve AT tobtu DOT com>

	This file is part of XSHA1 Reverser.

	XSHA1 Reverser is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	XSHA1 Reverser is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with XSHA1 Reverser.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef XSHA1_HELP_H
#define XSHA1_HELP_H

#include <stdio.h>
#include "common.h"

#ifndef ARC_x86
	// Non-x86... Yeah I got nothing
#else

#ifdef _WIN32
	#include <immintrin.h> // AVX and older
	#ifdef XOP_CODE
		#include <ammintrin.h> // XOP
	#endif
#else
	#include <x86intrin.h>
#endif

#if __XSHA1_INTERLACED__ < 1
	#error "__XSHA1_INTERLACED__ < 1"
#elif __XSHA1_INTERLACED__ > 2
	#error "__XSHA1_INTERLACED__ > 2"
#endif

#if defined(AVX_BROADCAST) && defined(AVX_CODE)
	#define LOAD_CONST_SHA1_A   _mm_broadcastd_epi32(0x67452301)
	#define LOAD_CONST_SHA1_B   _mm_broadcastd_epi32(0xefcdab89)
	#define LOAD_CONST_SHA1_C   _mm_broadcastd_epi32(0x98badcfe)
	#define LOAD_CONST_SHA1_D   _mm_broadcastd_epi32(0x10325476)
	#define LOAD_CONST_SHA1_E   _mm_broadcastd_epi32(0xc3d2e1f0)
	#define LOAD_CONST_SHA1_FF  _mm_broadcastd_epi32(0x5a827999)
	#define LOAD_CONST_SHA1_GG  _mm_broadcastd_epi32(0x6ed9eba1)
	#define LOAD_CONST_SHA1_HH  _mm_broadcastd_epi32(0x8f1bbcdc)
	#define LOAD_CONST_SHA1_II  _mm_broadcastd_epi32(0xca62c1d6)
#else
	#define LOAD_CONST_SHA1_A   _mm_set1_epi32(0x67452301)
	#define LOAD_CONST_SHA1_B   _mm_set1_epi32(0xefcdab89)
	#define LOAD_CONST_SHA1_C   _mm_set1_epi32(0x98badcfe)
	#define LOAD_CONST_SHA1_D   _mm_set1_epi32(0x10325476)
	#define LOAD_CONST_SHA1_E   _mm_set1_epi32(0xc3d2e1f0)
	#define LOAD_CONST_SHA1_FF  _mm_set1_epi32(0x5a827999)
	#define LOAD_CONST_SHA1_GG  _mm_set1_epi32(0x6ed9eba1)
	#define LOAD_CONST_SHA1_HH  _mm_set1_epi32(0x8f1bbcdc)
	#define LOAD_CONST_SHA1_II  _mm_set1_epi32(0xca62c1d6)
#endif

#define ZEROY(i,j)  _mm_set1_epi32(0)
#define y(i,j)      x[(i) * __XSHA1_INTERLACED__ + (j)]

#define SHA1_FF_REV_(a,b,c,d,e,x,i,j)      if (__XSHA1_INTERLACED__ > j) { b[j] = ROT(b[j], 2); e[j] = _mm_sub_epi32(_mm_sub_epi32(_mm_sub_epi32(_mm_sub_epi32(e[j], _mm_or_si128(_mm_and_si128(b[j], c[j]), _mm_andnot_si128(b[j], d[j]))                 ), x(i,j)), LOAD_CONST_SHA1_FF), ROT(a[j], 5)); }
//#define SHA1_FF_REV_(a,b,c,d,e,x,i,j)    if (__XSHA1_INTERLACED__ > j) { b[j] = ROT(b[j], 2); e[j] = _mm_sub_epi32(_mm_sub_epi32(_mm_sub_epi32(_mm_sub_epi32(e[j], _mm_xor_si128(_mm_and_si128(_mm_xor_si128(c[j], d[j]), b[j]), d[j])                   ), x(i,j)), LOAD_CONST_SHA1_FF), ROT(a[j], 5)); }
#define SHA1_GG_REV_(a,b,c,d,e,x,i,j)      if (__XSHA1_INTERLACED__ > j) { b[j] = ROT(b[j], 2); e[j] = _mm_sub_epi32(_mm_sub_epi32(_mm_sub_epi32(_mm_sub_epi32(e[j], _mm_xor_si128(b[j], _mm_xor_si128(c[j], d[j]))                                        ), x(i,j)), LOAD_CONST_SHA1_GG), ROT(a[j], 5)); }
#define SHA1_HH_REV_(a,b,c,d,e,x,i,j)      if (__XSHA1_INTERLACED__ > j) { b[j] = ROT(b[j], 2); e[j] = _mm_sub_epi32(_mm_sub_epi32(_mm_sub_epi32(_mm_sub_epi32(e[j], _mm_or_si128(_mm_and_si128(_mm_or_si128(b[j], c[j]), d[j]), _mm_and_si128(b[j], c[j]))), x(i,j)), LOAD_CONST_SHA1_HH), ROT(a[j], 5)); }
#define SHA1_II_REV_(a,b,c,d,e,x,i,j)      if (__XSHA1_INTERLACED__ > j) { b[j] = ROT(b[j], 2); e[j] = _mm_sub_epi32(_mm_sub_epi32(_mm_sub_epi32(_mm_sub_epi32(e[j], _mm_xor_si128(b[j], _mm_xor_si128(c[j], d[j]))                                        ), x(i,j)), LOAD_CONST_SHA1_II), ROT(a[j], 5)); }

#define SHA1_FF_REV(a,b,c,d,e,x,i)         SHA1_FF_REV_(  a,b,c,d,e,  x,i,0); SHA1_FF_REV_(  a,b,c,d,e,  x,i,1);
#define SHA1_GG_REV(a,b,c,d,e,x,i)         SHA1_GG_REV_(  a,b,c,d,e,  x,i,0); SHA1_GG_REV_(  a,b,c,d,e,  x,i,1);
#define SHA1_HH_REV(a,b,c,d,e,x,i)         SHA1_HH_REV_(  a,b,c,d,e,  x,i,0); SHA1_HH_REV_(  a,b,c,d,e,  x,i,1);
#define SHA1_II_REV(a,b,c,d,e,x,i)         SHA1_II_REV_(  a,b,c,d,e,  x,i,0); SHA1_II_REV_(  a,b,c,d,e,  x,i,1);

uint32_t finishRev16(__m128i *t, __m128i *a, __m128i *b, __m128i *c, uint32_t *sx, uint32_t x0, uint32_t x1, uint32_t x2, uint32_t x3, uint32_t &mask);
uint32_t finishRev20(__m128i *tmp, __m128i *t, __m128i *a, __m128i *b, __m128i *c, __m128i *d, uint32_t *sx, uint32_t x1, uint32_t x2, uint32_t x3, uint32_t x4, uint32_t &mask);

#define ROT(n,s) _mm_or_si128(_mm_slli_epi32(n, s), _mm_srli_epi32(n, (32 - (s))))

// The hard part in SSE2 is "1 << n"
// s - sign bit, e - exponent bit, f - fraction bit
// float is seeeeeeeefffffffffffffffffffffff (1 s, 8 e, 23 f)
// 1.0            is 0 01111111 00000000000000000000000 "0x3f800000"
// (n & 31) << 23 is 0 000nnnnn 00000000000000000000000
// 1.0 + ((n & 31) << 23) is "1.0 * pow(2, n)" or "(float) (1 << n)"
#define POW_2_OF_X(x) \
	/*      (int) *(      (float*) &(tmp)) */ \
	_mm_cvtps_epi32(_mm_castsi128_ps( \
		/*                    tmp = (x << 23) +               0x3f800000 */ \
		_mm_add_epi32(_mm_slli_epi32(x,   23), _mm_set1_epi32(0x3f800000))));
#define EXPANSION_(y,i,j)  if (__XSHA1_INTERLACED__ > j) { y(i+16,j) = POW_2_OF_X(_mm_and_si128(_mm_xor_si128(_mm_xor_si128(_mm_xor_si128(y(i,j), y(i+2,j)), y(i+8,j)), y(i+13,j)), _mm_set1_epi32(31))); }
#define EXPANSION(y,i)     EXPANSION_(y,i,0); EXPANSION_(y,i,1);

__inline__ uint32_t xsha1Reverse_16(const uint32_t *h, uint32_t &invalidCount, bool onlyValid, bool onlyPrintable, bool noCollisions, FILE *fout)
{
	__m128i a[__XSHA1_INTERLACED__], b[__XSHA1_INTERLACED__], c[__XSHA1_INTERLACED__], d[__XSHA1_INTERLACED__], e[__XSHA1_INTERLACED__], hash[5], y0, y1;
	__m128i t16a[__XSHA1_INTERLACED__], t16b[__XSHA1_INTERLACED__], t16c[__XSHA1_INTERLACED__], t16d[__XSHA1_INTERLACED__], t16e[__XSHA1_INTERLACED__];
	__m128i t17a[__XSHA1_INTERLACED__], t17b[__XSHA1_INTERLACED__], t17c[__XSHA1_INTERLACED__], t17d[__XSHA1_INTERLACED__], t17e[__XSHA1_INTERLACED__];
	__m128i x[80 * __XSHA1_INTERLACED__];
	uint32_t *sx = (uint32_t*) (x + 4 * __XSHA1_INTERLACED__), x0, x1, x2, x3, count = 0, stmp, invalid;

	invalidCount = 0;
	hash[0] = _mm_set1_epi32(h[0] - 0x67452301);
	hash[1] = _mm_set1_epi32(h[1] - 0xefcdab89);
	hash[2] = _mm_set1_epi32(h[2] - 0x98badcfe);
	hash[3] = _mm_set1_epi32(h[3] - 0x10325476);
	hash[4] = _mm_set1_epi32(h[4] - 0xc3d2e1f0);

	if (__XSHA1_INTERLACED__ > 0) { y(3,0) = _mm_set_epi32(3,2,1,0); }
	if (__XSHA1_INTERLACED__ > 1) { y(3,1) = _mm_set_epi32(7,6,5,4); }
#if __XSHA1_INTERLACED__ > 1
	for (x3 = 0; x3 < 32; x3 += 8, y(3,0) = _mm_add_epi32(y(3,0), _mm_set1_epi32(8)), y(3,1) = _mm_add_epi32(y(3,1), _mm_set1_epi32(8)))
#elif __XSHA1_INTERLACED__ > 0
	for (x3 = 0; x3 < 32; x3 += 4, y(3,0) = _mm_add_epi32(y(3,0), _mm_set1_epi32(4)))
#endif
	{
		y1 = _mm_set1_epi32(0);
		for (x1 = 0; x1 < 5; x1++, y1 = _mm_add_epi32(y1, _mm_set1_epi32(1)))
		{
			if (__XSHA1_INTERLACED__ > 0) { y(1,0) = _mm_xor_si128(y(3,0), y1); }
			if (__XSHA1_INTERLACED__ > 1) { y(1,1) = _mm_xor_si128(y(3,1), y1); }
			if (__XSHA1_INTERLACED__ > 0) { y(2,0) = _mm_set1_epi32(0); }
			if (__XSHA1_INTERLACED__ > 1) { y(2,1) = _mm_set1_epi32(0); }
#if __XSHA1_INTERLACED__ > 1
			for (x2 = 0; x2 < 32; x2++, y(2,1) = y(2,0) = _mm_add_epi32(y(2,0), _mm_set1_epi32(1)))
#elif __XSHA1_INTERLACED__ > 0
			for (x2 = 0; x2 < 32; x2++,          y(2,0) = _mm_add_epi32(y(2,0), _mm_set1_epi32(1)))
#endif
			{
				y0 = _mm_set1_epi32(0);
				for (x0 = 0; /*x0 < 5*/; x0++, y0 = _mm_add_epi32(y0, _mm_set1_epi32(1)))
				{
					if (__XSHA1_INTERLACED__ > 0) { y( 0,0) = _mm_xor_si128(y(2,0), y0); }
					if (__XSHA1_INTERLACED__ > 1) { y( 0,1) = y( 0,0); }
					if (__XSHA1_INTERLACED__ > 0) { y(16,0) = POW_2_OF_X(y0); }
					if (__XSHA1_INTERLACED__ > 1) { y(16,1) = y(16,0); }
					if (__XSHA1_INTERLACED__ > 0) { y(17,0) = POW_2_OF_X(y1); }
					if (__XSHA1_INTERLACED__ > 1) { y(17,1) = y(17,0); }
					if (__XSHA1_INTERLACED__ > 0) { y(18,0) = POW_2_OF_X(y(2,0)); }
					if (__XSHA1_INTERLACED__ > 1) { y(18,1) = y(18,0); }
					if (__XSHA1_INTERLACED__ > 0) { y(19,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(y( 3,0),                     y(16,0)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(19,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(y( 3,1),                     y(16,1)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 0) { y(20,0) = POW_2_OF_X(_mm_and_si128(                                                         y(17,0) , _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(20,1) = POW_2_OF_X(_mm_and_si128(                                                         y(17,1) , _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 0) { y(21,0) = POW_2_OF_X(_mm_and_si128(                                                         y(18,0) , _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(21,1) = POW_2_OF_X(_mm_and_si128(                                                         y(18,1) , _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 0) { y(22,0) = POW_2_OF_X(_mm_and_si128(                                                         y(19,0) , _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(22,1) = POW_2_OF_X(_mm_and_si128(                                                         y(19,1) , _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 0) { y(23,0) = POW_2_OF_X(_mm_and_si128(                                                         y(20,0) , _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(23,1) = POW_2_OF_X(_mm_and_si128(                                                         y(20,1) , _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 0) { y(24,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(16,0) , y(21,0)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(24,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(16,1) , y(21,1)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 0) { y(25,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(17,0) , y(22,0)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(25,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(17,1) , y(22,1)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 0) { y(26,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(18,0) , y(23,0)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(26,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(18,1) , y(23,1)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 0) { y(27,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(19,0) , y(24,0)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(27,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(19,1) , y(24,1)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 0) { y(28,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(20,0) , y(25,0)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(28,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(20,1) , y(25,1)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 0) { y(29,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(21,0) , y(26,0)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(29,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(21,1) , y(26,1)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 0) { y(30,0) = POW_2_OF_X(_mm_and_si128(_mm_xor_si128(_mm_xor_si128(         y(16,0) , y(22,0)), y(27,0)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(30,1) = POW_2_OF_X(_mm_and_si128(_mm_xor_si128(_mm_xor_si128(         y(16,1) , y(22,1)), y(27,1)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 0) { y(31,0) = POW_2_OF_X(_mm_and_si128(_mm_xor_si128(_mm_xor_si128(         y(17,0) , y(23,0)), y(28,0)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(31,1) = POW_2_OF_X(_mm_and_si128(_mm_xor_si128(_mm_xor_si128(         y(17,1) , y(23,1)), y(28,1)), _mm_set1_epi32(31))); }
					EXPANSION(y,16); EXPANSION(y,17); EXPANSION(y,18); EXPANSION(y,19);
					EXPANSION(y,20); EXPANSION(y,21); EXPANSION(y,22); EXPANSION(y,23);
					EXPANSION(y,24); EXPANSION(y,25); EXPANSION(y,26); EXPANSION(y,27);
					EXPANSION(y,28); EXPANSION(y,29); EXPANSION(y,30); EXPANSION(y,31);
					EXPANSION(y,32); EXPANSION(y,33); EXPANSION(y,34); EXPANSION(y,35);
					EXPANSION(y,36); EXPANSION(y,37); EXPANSION(y,38); EXPANSION(y,39);
					EXPANSION(y,40); EXPANSION(y,41); EXPANSION(y,42); EXPANSION(y,43);
					EXPANSION(y,44); EXPANSION(y,45); EXPANSION(y,46); EXPANSION(y,47);
					EXPANSION(y,48); EXPANSION(y,49); EXPANSION(y,50); EXPANSION(y,51);
					EXPANSION(y,52); EXPANSION(y,53); EXPANSION(y,54); EXPANSION(y,55);
					EXPANSION(y,56); EXPANSION(y,57); EXPANSION(y,58); EXPANSION(y,59);
					EXPANSION(y,60); EXPANSION(y,61); EXPANSION(y,62); EXPANSION(y,63);

					if (__XSHA1_INTERLACED__ > 0) { a[0] = hash[0]; }
					if (__XSHA1_INTERLACED__ > 1) { a[1] = hash[0]; }
					if (__XSHA1_INTERLACED__ > 0) { b[0] = hash[1]; }
					if (__XSHA1_INTERLACED__ > 1) { b[1] = hash[1]; }
					if (__XSHA1_INTERLACED__ > 0) { c[0] = hash[2]; }
					if (__XSHA1_INTERLACED__ > 1) { c[1] = hash[2]; }
					if (__XSHA1_INTERLACED__ > 0) { d[0] = hash[3]; }
					if (__XSHA1_INTERLACED__ > 1) { d[1] = hash[3]; }
					if (__XSHA1_INTERLACED__ > 0) { e[0] = hash[4]; }
					if (__XSHA1_INTERLACED__ > 1) { e[1] = hash[4]; }

					SHA1_II_REV(b,c,d,e,a,    y,79); SHA1_II_REV(c,d,e,a,b,    y,78); SHA1_II_REV(d,e,a,b,c,    y,77); SHA1_II_REV(e,a,b,c,d,    y,76); SHA1_II_REV(a,b,c,d,e,    y,75);
					SHA1_II_REV(b,c,d,e,a,    y,74); SHA1_II_REV(c,d,e,a,b,    y,73); SHA1_II_REV(d,e,a,b,c,    y,72); SHA1_II_REV(e,a,b,c,d,    y,71); SHA1_II_REV(a,b,c,d,e,    y,70);
					SHA1_II_REV(b,c,d,e,a,    y,69); SHA1_II_REV(c,d,e,a,b,    y,68); SHA1_II_REV(d,e,a,b,c,    y,67); SHA1_II_REV(e,a,b,c,d,    y,66); SHA1_II_REV(a,b,c,d,e,    y,65);
					SHA1_II_REV(b,c,d,e,a,    y,64); SHA1_II_REV(c,d,e,a,b,    y,63); SHA1_II_REV(d,e,a,b,c,    y,62); SHA1_II_REV(e,a,b,c,d,    y,61); SHA1_II_REV(a,b,c,d,e,    y,60);

					SHA1_HH_REV(b,c,d,e,a,    y,59); SHA1_HH_REV(c,d,e,a,b,    y,58); SHA1_HH_REV(d,e,a,b,c,    y,57); SHA1_HH_REV(e,a,b,c,d,    y,56); SHA1_HH_REV(a,b,c,d,e,    y,55);
					SHA1_HH_REV(b,c,d,e,a,    y,54); SHA1_HH_REV(c,d,e,a,b,    y,53); SHA1_HH_REV(d,e,a,b,c,    y,52); SHA1_HH_REV(e,a,b,c,d,    y,51); SHA1_HH_REV(a,b,c,d,e,    y,50);
					SHA1_HH_REV(b,c,d,e,a,    y,49); SHA1_HH_REV(c,d,e,a,b,    y,48); SHA1_HH_REV(d,e,a,b,c,    y,47); SHA1_HH_REV(e,a,b,c,d,    y,46); SHA1_HH_REV(a,b,c,d,e,    y,45);
					SHA1_HH_REV(b,c,d,e,a,    y,44); SHA1_HH_REV(c,d,e,a,b,    y,43); SHA1_HH_REV(d,e,a,b,c,    y,42); SHA1_HH_REV(e,a,b,c,d,    y,41); SHA1_HH_REV(a,b,c,d,e,    y,40);

					SHA1_GG_REV(b,c,d,e,a,    y,39); SHA1_GG_REV(c,d,e,a,b,    y,38); SHA1_GG_REV(d,e,a,b,c,    y,37); SHA1_GG_REV(e,a,b,c,d,    y,36); SHA1_GG_REV(a,b,c,d,e,    y,35);
					SHA1_GG_REV(b,c,d,e,a,    y,34); SHA1_GG_REV(c,d,e,a,b,    y,33); SHA1_GG_REV(d,e,a,b,c,    y,32); SHA1_GG_REV(e,a,b,c,d,    y,31); SHA1_GG_REV(a,b,c,d,e,    y,30);
					SHA1_GG_REV(b,c,d,e,a,    y,29); SHA1_GG_REV(c,d,e,a,b,    y,28); SHA1_GG_REV(d,e,a,b,c,    y,27); SHA1_GG_REV(e,a,b,c,d,    y,26); SHA1_GG_REV(a,b,c,d,e,    y,25);
					SHA1_GG_REV(b,c,d,e,a,    y,24); SHA1_GG_REV(c,d,e,a,b,    y,23); SHA1_GG_REV(d,e,a,b,c,    y,22); SHA1_GG_REV(e,a,b,c,d,    y,21); SHA1_GG_REV(a,b,c,d,e,    y,20);

					SHA1_FF_REV(b,c,d,e,a,    y,19); SHA1_FF_REV(c,d,e,a,b,    y,18); SHA1_FF_REV(d,e,a,b,c,    y,17);

					if (x0 == 5)
					{
						break;
					}

					SHA1_FF_REV(e,a,b,c,d,    y,16);

					SHA1_FF_REV(a,b,c,d,e,ZEROY,15);
					SHA1_FF_REV(b,c,d,e,a,ZEROY,14); SHA1_FF_REV(c,d,e,a,b,ZEROY,13); SHA1_FF_REV(d,e,a,b,c,ZEROY,12); SHA1_FF_REV(e,a,b,c,d,ZEROY,11); SHA1_FF_REV(a,b,c,d,e,ZEROY,10);
					SHA1_FF_REV(b,c,d,e,a,ZEROY, 9); SHA1_FF_REV(c,d,e,a,b,ZEROY, 8); SHA1_FF_REV(d,e,a,b,c,ZEROY, 7); SHA1_FF_REV(e,a,b,c,d,ZEROY, 6); SHA1_FF_REV(a,b,c,d,e,ZEROY, 5);
					SHA1_FF_REV(b,c,d,e,a,ZEROY, 4);

					if (__XSHA1_INTERLACED__ > 0)
					{
						stmp = _mm_movemask_epi8(_mm_cmpeq_epi32(a[0], _mm_set1_epi32(0x59d148c0)));
						while (stmp != 0)
						{
							if (finishRev16(b + 0, c + 0, d + 0, e + 0, sx, x0, x1, x2, x3, stmp))
							{
								invalid = outputPassword(sx, onlyValid, onlyPrintable, h, fout);
								invalidCount += invalid;
								count++;
								if (invalid == 0 && noCollisions != 0)
								{
									return count - invalidCount;
								}
							}
						}
					}
					if (__XSHA1_INTERLACED__ > 1)
					{
						stmp = _mm_movemask_epi8(_mm_cmpeq_epi32(a[1], _mm_set1_epi32(0x59d148c0)));
						while (stmp != 0)
						{
							if (finishRev16(b + 1, c + 1, d + 1, e + 1, sx, x0, x1, x2, x3 + 4, stmp))
							{
								invalid = outputPassword(sx, onlyValid, onlyPrintable, h, fout);
								invalidCount += invalid;
								count++;
								if (invalid == 0 && noCollisions != 0)
								{
									return count - invalidCount;
								}
							}
						}
					}
				}

				SHA1_FF_REV(e,a,b,c,d,ZEROY,16);
				if (__XSHA1_INTERLACED__ > 0) { t16a[0] = a[0]; }
				if (__XSHA1_INTERLACED__ > 1) { t16a[1] = a[1]; }
				if (__XSHA1_INTERLACED__ > 0) { t16b[0] = b[0]; }
				if (__XSHA1_INTERLACED__ > 1) { t16b[1] = b[1]; }
				if (__XSHA1_INTERLACED__ > 0) { t16c[0] = c[0]; }
				if (__XSHA1_INTERLACED__ > 1) { t16c[1] = c[1]; }
				if (__XSHA1_INTERLACED__ > 0) { t16d[0] = d[0]; }
				if (__XSHA1_INTERLACED__ > 1) { t16d[1] = d[1]; }
				if (__XSHA1_INTERLACED__ > 0) { t16e[0] = e[0]; }
				if (__XSHA1_INTERLACED__ > 1) { t16e[1] = e[1]; }
				// x[16] = (1 << 5) to (1 << 31)
#if __XSHA1_INTERLACED__ > 1
				for (; x0 < 32; x0++, y(16,1) = y(16,0) = _mm_slli_epi32(y(16,0), 1), y(0,1) = y(0,0) = _mm_xor_si128(y0 = _mm_add_epi32(y0, _mm_set1_epi32(1)), y(2,0)))
#elif __XSHA1_INTERLACED__ > 0
				for (; x0 < 32; x0++,           y(16,0) = _mm_slli_epi32(y(16,0), 1),          y(0,0) = _mm_xor_si128(y0 = _mm_add_epi32(y0, _mm_set1_epi32(1)), y(2,0)))
#endif
				{
					if (__XSHA1_INTERLACED__ > 0) { a[0] = t16a[0]; }
					if (__XSHA1_INTERLACED__ > 1) { a[1] = t16a[1]; }
					if (__XSHA1_INTERLACED__ > 0) { b[0] = t16b[0]; }
					if (__XSHA1_INTERLACED__ > 1) { b[1] = t16b[1]; }
					if (__XSHA1_INTERLACED__ > 0) { c[0] = t16c[0]; }
					if (__XSHA1_INTERLACED__ > 1) { c[1] = t16c[1]; }
					if (__XSHA1_INTERLACED__ > 0) { d[0] = _mm_sub_epi32(t16d[0], y(16,0)); }
					if (__XSHA1_INTERLACED__ > 1) { d[1] = _mm_sub_epi32(t16d[1], y(16,1)); }
					if (__XSHA1_INTERLACED__ > 0) { e[0] = t16e[0]; }
					if (__XSHA1_INTERLACED__ > 1) { e[1] = t16e[1]; }

					SHA1_FF_REV(a,b,c,d,e,ZEROY,15);
					SHA1_FF_REV(b,c,d,e,a,ZEROY,14); SHA1_FF_REV(c,d,e,a,b,ZEROY,13); SHA1_FF_REV(d,e,a,b,c,ZEROY,12); SHA1_FF_REV(e,a,b,c,d,ZEROY,11); SHA1_FF_REV(a,b,c,d,e,ZEROY,10);
					SHA1_FF_REV(b,c,d,e,a,ZEROY, 9); SHA1_FF_REV(c,d,e,a,b,ZEROY, 8); SHA1_FF_REV(d,e,a,b,c,ZEROY, 7); SHA1_FF_REV(e,a,b,c,d,ZEROY, 6); SHA1_FF_REV(a,b,c,d,e,ZEROY, 5);
					SHA1_FF_REV(b,c,d,e,a,ZEROY, 4);

					if (__XSHA1_INTERLACED__ > 0)
					{
						stmp = _mm_movemask_epi8(_mm_cmpeq_epi32(a[0], _mm_set1_epi32(0x59d148c0)));
						while (stmp != 0)
						{
							if (finishRev16(b + 0, c + 0, d + 0, e + 0, sx, x0, x1, x2, x3, stmp))
							{
								invalid = outputPassword(sx, onlyValid, onlyPrintable, h, fout);
								invalidCount += invalid;
								count++;
								if (invalid == 0 && noCollisions != 0)
								{
									return count - invalidCount;
								}
							}
						}
					}
					if (__XSHA1_INTERLACED__ > 1)
					{
						stmp = _mm_movemask_epi8(_mm_cmpeq_epi32(a[1], _mm_set1_epi32(0x59d148c0)));
						while (stmp != 0)
						{
							if (finishRev16(b + 1, c + 1, d + 1, e + 1, sx, x0, x1, x2, x3 + 4, stmp))
							{
								invalid = outputPassword(sx, onlyValid, onlyPrintable, h, fout);
								invalidCount += invalid;
								count++;
								if (invalid == 0 && noCollisions != 0)
								{
									return count - invalidCount;
								}
							}
						}
					}
				}
			}
		}
		if (__XSHA1_INTERLACED__ > 0) { y(1,0) = _mm_xor_si128(y(3,0), _mm_set1_epi32(5)); }
		if (__XSHA1_INTERLACED__ > 1) { y(1,1) = _mm_xor_si128(y(3,1), _mm_set1_epi32(5)); }
		if (__XSHA1_INTERLACED__ > 0) { y(2,0) = _mm_set1_epi32(0); }
		if (__XSHA1_INTERLACED__ > 1) { y(2,1) = _mm_set1_epi32(0); }
#if __XSHA1_INTERLACED__ > 1
		for (x2 = 0; x2 < 32; x2++, y(2,1) = y(2,0) = _mm_add_epi32(y(2,0), _mm_set1_epi32(1)))
#elif __XSHA1_INTERLACED__ > 0
		for (x2 = 0; x2 < 32; x2++,          y(2,0) = _mm_add_epi32(y(2,0), _mm_set1_epi32(1)))
#endif
		{
			y0 = _mm_set1_epi32(0);
			for (x0 = 0; /*x0 < 5*/; x0++, y0 = _mm_add_epi32(y0, _mm_set1_epi32(1)))
			{
				if (__XSHA1_INTERLACED__ > 0) { y( 0,0) = _mm_xor_si128(y(2,0), y0); }
				if (__XSHA1_INTERLACED__ > 1) { y( 0,1) = y( 0,0); }
				if (__XSHA1_INTERLACED__ > 0) { y(16,0) = POW_2_OF_X(y0); }
				if (__XSHA1_INTERLACED__ > 1) { y(16,1) = y(16,0); }
				if (__XSHA1_INTERLACED__ > 0) { y(17,0) = _mm_set1_epi32(1 << 5); }
				if (__XSHA1_INTERLACED__ > 1) { y(17,1) = _mm_set1_epi32(1 << 5); }
				if (__XSHA1_INTERLACED__ > 0) { y(18,0) = POW_2_OF_X(y(2,0)); }
				if (__XSHA1_INTERLACED__ > 1) { y(18,1) = y(18,0); }
				if (__XSHA1_INTERLACED__ > 0) { y(19,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(y( 3,0),                     y(16,0)), _mm_set1_epi32(31))); }
				if (__XSHA1_INTERLACED__ > 1) { y(19,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(y( 3,1),                     y(16,1)), _mm_set1_epi32(31))); }
				if (__XSHA1_INTERLACED__ > 0) { y(20,0) = _mm_set1_epi32(1); }
				if (__XSHA1_INTERLACED__ > 1) { y(20,1) = _mm_set1_epi32(1); }
				if (__XSHA1_INTERLACED__ > 0) { y(21,0) = POW_2_OF_X(_mm_and_si128(                                                         y(18,0) , _mm_set1_epi32(31))); }
				if (__XSHA1_INTERLACED__ > 1) { y(21,1) = POW_2_OF_X(_mm_and_si128(                                                         y(18,1) , _mm_set1_epi32(31))); }
				if (__XSHA1_INTERLACED__ > 0) { y(22,0) = POW_2_OF_X(_mm_and_si128(                                                         y(19,0) , _mm_set1_epi32(31))); }
				if (__XSHA1_INTERLACED__ > 1) { y(22,1) = POW_2_OF_X(_mm_and_si128(                                                         y(19,1) , _mm_set1_epi32(31))); }
				if (__XSHA1_INTERLACED__ > 0) { y(23,0) = POW_2_OF_X(_mm_and_si128(                                                         y(20,0) , _mm_set1_epi32(31))); }
				if (__XSHA1_INTERLACED__ > 1) { y(23,1) = POW_2_OF_X(_mm_and_si128(                                                         y(20,1) , _mm_set1_epi32(31))); }
				if (__XSHA1_INTERLACED__ > 0) { y(24,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(16,0) , y(21,0)), _mm_set1_epi32(31))); }
				if (__XSHA1_INTERLACED__ > 1) { y(24,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(16,1) , y(21,1)), _mm_set1_epi32(31))); }
				if (__XSHA1_INTERLACED__ > 0) { y(25,0) = POW_2_OF_X(_mm_and_si128(                                                         y(22,0) , _mm_set1_epi32(31))); }
				if (__XSHA1_INTERLACED__ > 1) { y(25,1) = POW_2_OF_X(_mm_and_si128(                                                         y(22,1) , _mm_set1_epi32(31))); }
				if (__XSHA1_INTERLACED__ > 0) { y(26,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(18,0) , y(23,0)), _mm_set1_epi32(31))); }
				if (__XSHA1_INTERLACED__ > 1) { y(26,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(18,1) , y(23,1)), _mm_set1_epi32(31))); }
				if (__XSHA1_INTERLACED__ > 0) { y(27,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(19,0) , y(24,0)), _mm_set1_epi32(31))); }
				if (__XSHA1_INTERLACED__ > 1) { y(27,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(19,1) , y(24,1)), _mm_set1_epi32(31))); }
				if (__XSHA1_INTERLACED__ > 0) { y(28,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(20,0) , y(25,0)), _mm_set1_epi32(31))); }
				if (__XSHA1_INTERLACED__ > 1) { y(28,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(20,1) , y(25,1)), _mm_set1_epi32(31))); }
				if (__XSHA1_INTERLACED__ > 0) { y(29,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(21,0) , y(26,0)), _mm_set1_epi32(31))); }
				if (__XSHA1_INTERLACED__ > 1) { y(29,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(21,1) , y(26,1)), _mm_set1_epi32(31))); }
				if (__XSHA1_INTERLACED__ > 0) { y(30,0) = POW_2_OF_X(_mm_and_si128(_mm_xor_si128(_mm_xor_si128(         y(16,0) , y(22,0)), y(27,0)), _mm_set1_epi32(31))); }
				if (__XSHA1_INTERLACED__ > 1) { y(30,1) = POW_2_OF_X(_mm_and_si128(_mm_xor_si128(_mm_xor_si128(         y(16,1) , y(22,1)), y(27,1)), _mm_set1_epi32(31))); }
				if (__XSHA1_INTERLACED__ > 0) { y(31,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(23,0) , y(28,0)), _mm_set1_epi32(31))); }
				if (__XSHA1_INTERLACED__ > 1) { y(31,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(23,1) , y(28,1)), _mm_set1_epi32(31))); }
				EXPANSION(y,16); EXPANSION(y,17); EXPANSION(y,18); EXPANSION(y,19);
				EXPANSION(y,20); EXPANSION(y,21); EXPANSION(y,22); EXPANSION(y,23);
				EXPANSION(y,24); EXPANSION(y,25); EXPANSION(y,26); EXPANSION(y,27);
				EXPANSION(y,28); EXPANSION(y,29); EXPANSION(y,30); EXPANSION(y,31);
				EXPANSION(y,32); EXPANSION(y,33); EXPANSION(y,34); EXPANSION(y,35);
				EXPANSION(y,36); EXPANSION(y,37); EXPANSION(y,38); EXPANSION(y,39);
				EXPANSION(y,40); EXPANSION(y,41); EXPANSION(y,42); EXPANSION(y,43);
				EXPANSION(y,44); EXPANSION(y,45); EXPANSION(y,46); EXPANSION(y,47);
				EXPANSION(y,48); EXPANSION(y,49); EXPANSION(y,50); EXPANSION(y,51);
				EXPANSION(y,52); EXPANSION(y,53); EXPANSION(y,54); EXPANSION(y,55);
				EXPANSION(y,56); EXPANSION(y,57); EXPANSION(y,58); EXPANSION(y,59);
				EXPANSION(y,60); EXPANSION(y,61); EXPANSION(y,62); EXPANSION(y,63);

				if (__XSHA1_INTERLACED__ > 0) { a[0] = hash[0]; }
				if (__XSHA1_INTERLACED__ > 1) { a[1] = hash[0]; }
				if (__XSHA1_INTERLACED__ > 0) { b[0] = hash[1]; }
				if (__XSHA1_INTERLACED__ > 1) { b[1] = hash[1]; }
				if (__XSHA1_INTERLACED__ > 0) { c[0] = hash[2]; }
				if (__XSHA1_INTERLACED__ > 1) { c[1] = hash[2]; }
				if (__XSHA1_INTERLACED__ > 0) { d[0] = hash[3]; }
				if (__XSHA1_INTERLACED__ > 1) { d[1] = hash[3]; }
				if (__XSHA1_INTERLACED__ > 0) { e[0] = hash[4]; }
				if (__XSHA1_INTERLACED__ > 1) { e[1] = hash[4]; }

				SHA1_II_REV(b,c,d,e,a,    y,79); SHA1_II_REV(c,d,e,a,b,    y,78); SHA1_II_REV(d,e,a,b,c,    y,77); SHA1_II_REV(e,a,b,c,d,    y,76); SHA1_II_REV(a,b,c,d,e,    y,75);
				SHA1_II_REV(b,c,d,e,a,    y,74); SHA1_II_REV(c,d,e,a,b,    y,73); SHA1_II_REV(d,e,a,b,c,    y,72); SHA1_II_REV(e,a,b,c,d,    y,71); SHA1_II_REV(a,b,c,d,e,    y,70);
				SHA1_II_REV(b,c,d,e,a,    y,69); SHA1_II_REV(c,d,e,a,b,    y,68); SHA1_II_REV(d,e,a,b,c,    y,67); SHA1_II_REV(e,a,b,c,d,    y,66); SHA1_II_REV(a,b,c,d,e,    y,65);
				SHA1_II_REV(b,c,d,e,a,    y,64); SHA1_II_REV(c,d,e,a,b,    y,63); SHA1_II_REV(d,e,a,b,c,    y,62); SHA1_II_REV(e,a,b,c,d,    y,61); SHA1_II_REV(a,b,c,d,e,    y,60);

				SHA1_HH_REV(b,c,d,e,a,    y,59); SHA1_HH_REV(c,d,e,a,b,    y,58); SHA1_HH_REV(d,e,a,b,c,    y,57); SHA1_HH_REV(e,a,b,c,d,    y,56); SHA1_HH_REV(a,b,c,d,e,    y,55);
				SHA1_HH_REV(b,c,d,e,a,    y,54); SHA1_HH_REV(c,d,e,a,b,    y,53); SHA1_HH_REV(d,e,a,b,c,    y,52); SHA1_HH_REV(e,a,b,c,d,    y,51); SHA1_HH_REV(a,b,c,d,e,    y,50);
				SHA1_HH_REV(b,c,d,e,a,    y,49); SHA1_HH_REV(c,d,e,a,b,    y,48); SHA1_HH_REV(d,e,a,b,c,    y,47); SHA1_HH_REV(e,a,b,c,d,    y,46); SHA1_HH_REV(a,b,c,d,e,    y,45);
				SHA1_HH_REV(b,c,d,e,a,    y,44); SHA1_HH_REV(c,d,e,a,b,    y,43); SHA1_HH_REV(d,e,a,b,c,    y,42); SHA1_HH_REV(e,a,b,c,d,    y,41); SHA1_HH_REV(a,b,c,d,e,    y,40);

				SHA1_GG_REV(b,c,d,e,a,    y,39); SHA1_GG_REV(c,d,e,a,b,    y,38); SHA1_GG_REV(d,e,a,b,c,    y,37); SHA1_GG_REV(e,a,b,c,d,    y,36); SHA1_GG_REV(a,b,c,d,e,    y,35);
				SHA1_GG_REV(b,c,d,e,a,    y,34); SHA1_GG_REV(c,d,e,a,b,    y,33); SHA1_GG_REV(d,e,a,b,c,    y,32); SHA1_GG_REV(e,a,b,c,d,    y,31); SHA1_GG_REV(a,b,c,d,e,    y,30);
				SHA1_GG_REV(b,c,d,e,a,    y,29); SHA1_GG_REV(c,d,e,a,b,    y,28); SHA1_GG_REV(d,e,a,b,c,    y,27); SHA1_GG_REV(e,a,b,c,d,    y,26); SHA1_GG_REV(a,b,c,d,e,    y,25);
				SHA1_GG_REV(b,c,d,e,a,    y,24); SHA1_GG_REV(c,d,e,a,b,    y,23); SHA1_GG_REV(d,e,a,b,c,    y,22); SHA1_GG_REV(e,a,b,c,d,    y,21); SHA1_GG_REV(a,b,c,d,e,    y,20);

				SHA1_FF_REV(b,c,d,e,a,    y,19); SHA1_FF_REV(c,d,e,a,b,    y,18);

				SHA1_FF_REV(d,e,a,b,c,ZEROY,17);
				if (__XSHA1_INTERLACED__ > 0) { t17a[0] = a[0]; }
				if (__XSHA1_INTERLACED__ > 1) { t17a[1] = a[1]; }
				if (__XSHA1_INTERLACED__ > 0) { t17b[0] = b[0]; }
				if (__XSHA1_INTERLACED__ > 1) { t17b[1] = b[1]; }
				if (__XSHA1_INTERLACED__ > 0) { t17c[0] = c[0]; }
				if (__XSHA1_INTERLACED__ > 1) { t17c[1] = c[1]; }
				if (__XSHA1_INTERLACED__ > 0) { t17d[0] = d[0]; }
				if (__XSHA1_INTERLACED__ > 1) { t17d[1] = d[1]; }
				if (__XSHA1_INTERLACED__ > 0) { t17e[0] = e[0]; }
				if (__XSHA1_INTERLACED__ > 1) { t17e[1] = e[1]; }

				if (x0 == 5)
				{
					break;
				}

				// x[16] = (1 << 5) to (1 << 31)
#if __XSHA1_INTERLACED__ > 1
				for (; x1 < 32; x1++, y(17,1) = y(17,0) = _mm_slli_epi32(y(17,0), 1), y(1,1) = y(1,0) = _mm_xor_si128(y1 = _mm_add_epi32(y1, _mm_set1_epi32(1)), y(3,0)))
#elif __XSHA1_INTERLACED__ > 0
				for (; x1 < 32; x1++,           y(17,0) = _mm_slli_epi32(y(17,0), 1),          y(1,0) = _mm_xor_si128(y1 = _mm_add_epi32(y1, _mm_set1_epi32(1)), y(3,0)))
#endif
				{
					if (__XSHA1_INTERLACED__ > 0) { a[0] = t17a[0]; }
					if (__XSHA1_INTERLACED__ > 1) { a[1] = t17a[1]; }
					if (__XSHA1_INTERLACED__ > 0) { b[0] = t17b[0]; }
					if (__XSHA1_INTERLACED__ > 1) { b[1] = t17b[1]; }
					if (__XSHA1_INTERLACED__ > 0) { c[0] = _mm_sub_epi32(t17c[0], y(17,0)); }
					if (__XSHA1_INTERLACED__ > 1) { c[1] = _mm_sub_epi32(t17c[1], y(17,1)); }
					if (__XSHA1_INTERLACED__ > 0) { d[0] = t17d[0]; }
					if (__XSHA1_INTERLACED__ > 1) { d[1] = t17d[1]; }
					if (__XSHA1_INTERLACED__ > 0) { e[0] = t17e[0]; }
					if (__XSHA1_INTERLACED__ > 1) { e[1] = t17e[1]; }

					SHA1_FF_REV(e,a,b,c,d,    y,16);

					SHA1_FF_REV(a,b,c,d,e,ZEROY,15);
					SHA1_FF_REV(b,c,d,e,a,ZEROY,14); SHA1_FF_REV(c,d,e,a,b,ZEROY,13); SHA1_FF_REV(d,e,a,b,c,ZEROY,12); SHA1_FF_REV(e,a,b,c,d,ZEROY,11); SHA1_FF_REV(a,b,c,d,e,ZEROY,10);
					SHA1_FF_REV(b,c,d,e,a,ZEROY, 9); SHA1_FF_REV(c,d,e,a,b,ZEROY, 8); SHA1_FF_REV(d,e,a,b,c,ZEROY, 7); SHA1_FF_REV(e,a,b,c,d,ZEROY, 6); SHA1_FF_REV(a,b,c,d,e,ZEROY, 5);
					SHA1_FF_REV(b,c,d,e,a,ZEROY, 4);

					if (__XSHA1_INTERLACED__ > 0)
					{
						stmp = _mm_movemask_epi8(_mm_cmpeq_epi32(a[0], _mm_set1_epi32(0x59d148c0)));
						while (stmp != 0)
						{
							if (finishRev16(b + 0, c + 0, d + 0, e + 0, sx, x0, x1, x2, x3, stmp))
							{
								invalid = outputPassword(sx, onlyValid, onlyPrintable, h, fout);
								invalidCount += invalid;
								count++;
								if (invalid == 0 && noCollisions != 0)
								{
									return count - invalidCount;
								}
							}
						}
					}
					if (__XSHA1_INTERLACED__ > 1)
					{
						stmp = _mm_movemask_epi8(_mm_cmpeq_epi32(a[1], _mm_set1_epi32(0x59d148c0)));
						while (stmp != 0)
						{
							if (finishRev16(b + 1, c + 1, d + 1, e + 1, sx, x0, x1, x2, x3 + 4, stmp))
							{
								invalid = outputPassword(sx, onlyValid, onlyPrintable, h, fout);
								invalidCount += invalid;
								count++;
								if (invalid == 0 && noCollisions != 0)
								{
									return count - invalidCount;
								}
							}
						}
					}
				}
				x1 = 5;
				y1 = _mm_set1_epi32(5);
				if (__XSHA1_INTERLACED__ > 0) { y(1,0) = _mm_xor_si128(y(3,0), _mm_set1_epi32(5)); }
				if (__XSHA1_INTERLACED__ > 1) { y(1,1) = _mm_xor_si128(y(3,1), _mm_set1_epi32(5)); }
			}

			// x[17] = (1 << 5) to (1 << 31)
#if __XSHA1_INTERLACED__ > 1
			for (; x1 < 32; x1++, y(17,1) = y(17,0) = _mm_slli_epi32(y(17,0), 1), y(1,1) = y(1,0) = _mm_xor_si128(y1 = _mm_add_epi32(y1, _mm_set1_epi32(1)), y(3,0)))
#elif __XSHA1_INTERLACED__ > 0
			for (; x1 < 32; x1++,           y(17,0) = _mm_slli_epi32(y(17,0), 1),          y(1,0) = _mm_xor_si128(y1 = _mm_add_epi32(y1, _mm_set1_epi32(1)), y(3,0)))
#endif
			{
				if (__XSHA1_INTERLACED__ > 0) { a[0] = t17a[0]; }
				if (__XSHA1_INTERLACED__ > 1) { a[1] = t17a[1]; }
				if (__XSHA1_INTERLACED__ > 0) { b[0] = t17b[0]; }
				if (__XSHA1_INTERLACED__ > 1) { b[1] = t17b[1]; }
				if (__XSHA1_INTERLACED__ > 0) { c[0] = _mm_sub_epi32(t17c[0], y(17,0)); }
				if (__XSHA1_INTERLACED__ > 1) { c[1] = _mm_sub_epi32(t17c[1], y(17,1)); }
				if (__XSHA1_INTERLACED__ > 0) { d[0] = t17d[0]; }
				if (__XSHA1_INTERLACED__ > 1) { d[1] = t17d[1]; }
				if (__XSHA1_INTERLACED__ > 0) { e[0] = t17e[0]; }
				if (__XSHA1_INTERLACED__ > 1) { e[1] = t17e[1]; }

				SHA1_FF_REV(e,a,b,c,d,ZEROY,16);
				if (__XSHA1_INTERLACED__ > 0) { t16a[0] = a[0]; }
				if (__XSHA1_INTERLACED__ > 1) { t16a[1] = a[1]; }
				if (__XSHA1_INTERLACED__ > 0) { t16b[0] = b[0]; }
				if (__XSHA1_INTERLACED__ > 1) { t16b[1] = b[1]; }
				if (__XSHA1_INTERLACED__ > 0) { t16c[0] = c[0]; }
				if (__XSHA1_INTERLACED__ > 1) { t16c[1] = c[1]; }
				if (__XSHA1_INTERLACED__ > 0) { t16d[0] = d[0]; }
				if (__XSHA1_INTERLACED__ > 1) { t16d[1] = d[1]; }
				if (__XSHA1_INTERLACED__ > 0) { t16e[0] = e[0]; }
				if (__XSHA1_INTERLACED__ > 1) { t16e[1] = e[1]; }
				// x[16] = (1 << 5) to (1 << 31)
#if __XSHA1_INTERLACED__ > 1
				for (; x0 < 32; x0++, y(16,1) = y(16,0) = _mm_slli_epi32(y(16,0), 1), y(0,1) = y(0,0) = _mm_xor_si128(y0 = _mm_add_epi32(y0, _mm_set1_epi32(1)), y(2,0)))
#elif __XSHA1_INTERLACED__ > 0
				for (; x0 < 32; x0++,           y(16,0) = _mm_slli_epi32(y(16,0), 1),          y(0,0) = _mm_xor_si128(y0 = _mm_add_epi32(y0, _mm_set1_epi32(1)), y(2,0)))
#endif
				{
					if (__XSHA1_INTERLACED__ > 0) { a[0] = t16a[0]; }
					if (__XSHA1_INTERLACED__ > 1) { a[1] = t16a[1]; }
					if (__XSHA1_INTERLACED__ > 0) { b[0] = t16b[0]; }
					if (__XSHA1_INTERLACED__ > 1) { b[1] = t16b[1]; }
					if (__XSHA1_INTERLACED__ > 0) { c[0] = t16c[0]; }
					if (__XSHA1_INTERLACED__ > 1) { c[1] = t16c[1]; }
					if (__XSHA1_INTERLACED__ > 0) { d[0] = _mm_sub_epi32(t16d[0], y(16,0)); }
					if (__XSHA1_INTERLACED__ > 1) { d[1] = _mm_sub_epi32(t16d[1], y(16,1)); }
					if (__XSHA1_INTERLACED__ > 0) { e[0] = t16e[0]; }
					if (__XSHA1_INTERLACED__ > 1) { e[1] = t16e[1]; }

					SHA1_FF_REV(a,b,c,d,e,ZEROY,15);
					SHA1_FF_REV(b,c,d,e,a,ZEROY,14); SHA1_FF_REV(c,d,e,a,b,ZEROY,13); SHA1_FF_REV(d,e,a,b,c,ZEROY,12); SHA1_FF_REV(e,a,b,c,d,ZEROY,11); SHA1_FF_REV(a,b,c,d,e,ZEROY,10);
					SHA1_FF_REV(b,c,d,e,a,ZEROY, 9); SHA1_FF_REV(c,d,e,a,b,ZEROY, 8); SHA1_FF_REV(d,e,a,b,c,ZEROY, 7); SHA1_FF_REV(e,a,b,c,d,ZEROY, 6); SHA1_FF_REV(a,b,c,d,e,ZEROY, 5);
					SHA1_FF_REV(b,c,d,e,a,ZEROY, 4);

					if (__XSHA1_INTERLACED__ > 0)
					{
						stmp = _mm_movemask_epi8(_mm_cmpeq_epi32(a[0], _mm_set1_epi32(0x59d148c0)));
						while (stmp != 0)
						{
							if (finishRev16(b + 0, c + 0, d + 0, e + 0, sx, x0, x1, x2, x3, stmp))
							{
								invalid = outputPassword(sx, onlyValid, onlyPrintable, h, fout);
								invalidCount += invalid;
								count++;
								if (invalid == 0 && noCollisions != 0)
								{
									return count - invalidCount;
								}
							}
						}
					}
					if (__XSHA1_INTERLACED__ > 1)
					{
						stmp = _mm_movemask_epi8(_mm_cmpeq_epi32(a[1], _mm_set1_epi32(0x59d148c0)));
						while (stmp != 0)
						{
							if (finishRev16(b + 1, c + 1, d + 1, e + 1, sx, x0, x1, x2, x3 + 4, stmp))
							{
								invalid = outputPassword(sx, onlyValid, onlyPrintable, h, fout);
								invalidCount += invalid;
								count++;
								if (invalid == 0 && noCollisions != 0)
								{
									return count - invalidCount;
								}
							}
						}
					}
				}
				x0 = 5;
				y0 = _mm_set1_epi32(5);
				if (__XSHA1_INTERLACED__ > 0) { y( 0,0) = _mm_xor_si128(y(2,0), _mm_set1_epi32(5)); }
				if (__XSHA1_INTERLACED__ > 1) { y( 0,1) = _mm_xor_si128(y(2,1), _mm_set1_epi32(5)); }
				if (__XSHA1_INTERLACED__ > 0) { y(16,0) = _mm_set1_epi32(1 << 5); }
				if (__XSHA1_INTERLACED__ > 1) { y(16,1) = _mm_set1_epi32(1 << 5); }
			}
			x1 = 5;
			y1 = _mm_set1_epi32(5);
			if (__XSHA1_INTERLACED__ > 0) { y(1,0) = _mm_xor_si128(y(3,0), _mm_set1_epi32(5)); }
			if (__XSHA1_INTERLACED__ > 1) { y(1,1) = _mm_xor_si128(y(3,1), _mm_set1_epi32(5)); }
		}
	}
	return count - invalidCount;
}
#undef ROT

#define ROT(n,s) _mm_or_si128(_mm_slli_epi32(n, s), _mm_srli_epi32(n, (32 - (s))))
__inline__ uint32_t xsha1Reverse_20(const uint32_t *h, uint32_t &invalidCount, bool onlyValid, bool onlyPrintable, bool noCollisions, FILE *fout)
{
	__m128i a[__XSHA1_INTERLACED__], b[__XSHA1_INTERLACED__], c[__XSHA1_INTERLACED__], d[__XSHA1_INTERLACED__], e[__XSHA1_INTERLACED__], tmp, hash[5], y0, y1;
	__m128i t16a[__XSHA1_INTERLACED__], t16b[__XSHA1_INTERLACED__], t16c[__XSHA1_INTERLACED__], t16d[__XSHA1_INTERLACED__], t16e[__XSHA1_INTERLACED__];
	__m128i t17a[__XSHA1_INTERLACED__], t17b[__XSHA1_INTERLACED__], t17c[__XSHA1_INTERLACED__], t17d[__XSHA1_INTERLACED__], t17e[__XSHA1_INTERLACED__];
	__m128i x[80 * __XSHA1_INTERLACED__];
	uint32_t *sx = (uint32_t*) (x + 5 * __XSHA1_INTERLACED__), x0, x1, x2, x3, x4, count = 0, stmp, invalid;

	invalidCount = 0;
	hash[0] = _mm_set1_epi32(h[0] - 0x67452301);
	hash[1] = _mm_set1_epi32(h[1] - 0xefcdab89);
	hash[2] = _mm_set1_epi32(h[2] - 0x98badcfe);
	hash[3] = _mm_set1_epi32(h[3] - 0x10325476);
	hash[4] = _mm_set1_epi32(h[4] - 0xc3d2e1f0);

	if (__XSHA1_INTERLACED__ > 0) { y(4,0) = _mm_set_epi32(3,2,1,0); }
	if (__XSHA1_INTERLACED__ > 1) { y(4,1) = _mm_set_epi32(7,6,5,4); }
#if __XSHA1_INTERLACED__ > 1
	for (x4 = 0; x4 < 32; x4 += 8, y(4,0) = _mm_add_epi32(y(4,0), _mm_set1_epi32(8)), y(4,1) = _mm_add_epi32(y(4,1), _mm_set1_epi32(8)))
#elif __XSHA1_INTERLACED__ > 0
	for (x4 = 0; x4 < 32; x4 += 4, y(4,0) = _mm_add_epi32(y(4,0), _mm_set1_epi32(4)))
#endif
	{
		if (__XSHA1_INTERLACED__ > 0) { y(3,0) = _mm_set1_epi32(0); }
		if (__XSHA1_INTERLACED__ > 1) { y(3,1) = _mm_set1_epi32(0); }
#if __XSHA1_INTERLACED__ > 1
		for (x3 = 0; x3 < 32; x3++, y(3,0) = _mm_add_epi32(y(3,0), _mm_set1_epi32(1)), y(3,1) = _mm_add_epi32(y(3,1), _mm_set1_epi32(1)))
#elif __XSHA1_INTERLACED__ > 0
		for (x3 = 0; x3 < 32; x3++, y(3,0) = _mm_add_epi32(y(3,0), _mm_set1_epi32(1)))
#endif
		{
			y1 = _mm_set1_epi32(0);
			for (x1 = 0; x1 < 5; x1++, y1 = _mm_add_epi32(y1, _mm_set1_epi32(1)))
			{
				if (__XSHA1_INTERLACED__ > 0) { y(1,0) = _mm_xor_si128(y(3,0), y1); }
				if (__XSHA1_INTERLACED__ > 1) { y(1,1) = _mm_xor_si128(y(3,1), y1); }
				if (__XSHA1_INTERLACED__ > 0) { y(2,0) = _mm_set1_epi32(0); }
				if (__XSHA1_INTERLACED__ > 1) { y(2,1) = _mm_set1_epi32(0); }
#if __XSHA1_INTERLACED__ > 1
				for (x2 = 0; x2 < 32; x2++, y(2,1) = y(2,0) = _mm_add_epi32(y(2,0), _mm_set1_epi32(1)))
#elif __XSHA1_INTERLACED__ > 0
				for (x2 = 0; x2 < 32; x2++,          y(2,0) = _mm_add_epi32(y(2,0), _mm_set1_epi32(1)))
#endif
				{
					y0 = _mm_set1_epi32(0);
					for (x0 = 0; /*x0 < 5*/; x0++, y0 = _mm_add_epi32(y0, _mm_set1_epi32(1)))
					{
						if (__XSHA1_INTERLACED__ > 0) { y( 0,0) = _mm_xor_si128(y(2,0), y0); }
						if (__XSHA1_INTERLACED__ > 1) { y( 0,1) = y( 0,0); }
						if (__XSHA1_INTERLACED__ > 0) { y(16,0) = POW_2_OF_X(y0); }
						if (__XSHA1_INTERLACED__ > 1) { y(16,1) = y(16,0); }
						if (__XSHA1_INTERLACED__ > 0) { y(17,0) = POW_2_OF_X(y1); }
						if (__XSHA1_INTERLACED__ > 1) { y(17,1) = y(17,0); }
						if (__XSHA1_INTERLACED__ > 0) { y(18,0) = POW_2_OF_X(                            _mm_xor_si128(y( 2,0), y( 4,0))); }
						if (__XSHA1_INTERLACED__ > 1) { y(18,1) = POW_2_OF_X(                            _mm_xor_si128(y( 2,1), y( 4,1))); }
						if (__XSHA1_INTERLACED__ > 0) { y(19,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(y( 3,0),                     y(16,0)), _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 1) { y(19,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(y( 3,1),                     y(16,1)), _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 0) { y(20,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(y( 4,0),                     y(17,0)), _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 1) { y(20,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(y( 4,1),                     y(17,1)), _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 0) { y(21,0) = POW_2_OF_X(_mm_and_si128(                                                         y(18,0) , _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 1) { y(21,1) = POW_2_OF_X(_mm_and_si128(                                                         y(18,1) , _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 0) { y(22,0) = POW_2_OF_X(_mm_and_si128(                                                         y(19,0) , _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 1) { y(22,1) = POW_2_OF_X(_mm_and_si128(                                                         y(19,1) , _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 0) { y(23,0) = POW_2_OF_X(_mm_and_si128(                                                         y(20,0) , _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 1) { y(23,1) = POW_2_OF_X(_mm_and_si128(                                                         y(20,1) , _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 0) { y(24,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(16,0) , y(21,0)), _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 1) { y(24,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(16,1) , y(21,1)), _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 0) { y(25,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(17,0) , y(22,0)), _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 1) { y(25,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(17,1) , y(22,1)), _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 0) { y(26,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(18,0) , y(23,0)), _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 1) { y(26,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(18,1) , y(23,1)), _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 0) { y(27,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(19,0) , y(24,0)), _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 1) { y(27,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(19,1) , y(24,1)), _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 0) { y(28,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(20,0) , y(25,0)), _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 1) { y(28,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(20,1) , y(25,1)), _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 0) { y(29,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(21,0) , y(26,0)), _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 1) { y(29,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(21,1) , y(26,1)), _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 0) { y(30,0) = POW_2_OF_X(_mm_and_si128(_mm_xor_si128(_mm_xor_si128(         y(16,0) , y(22,0)), y(27,0)), _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 1) { y(30,1) = POW_2_OF_X(_mm_and_si128(_mm_xor_si128(_mm_xor_si128(         y(16,1) , y(22,1)), y(27,1)), _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 0) { y(31,0) = POW_2_OF_X(_mm_and_si128(_mm_xor_si128(_mm_xor_si128(         y(17,0) , y(23,0)), y(28,0)), _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 1) { y(31,1) = POW_2_OF_X(_mm_and_si128(_mm_xor_si128(_mm_xor_si128(         y(17,1) , y(23,1)), y(28,1)), _mm_set1_epi32(31))); }
						EXPANSION(y,16); EXPANSION(y,17); EXPANSION(y,18); EXPANSION(y,19);
						EXPANSION(y,20); EXPANSION(y,21); EXPANSION(y,22); EXPANSION(y,23);
						EXPANSION(y,24); EXPANSION(y,25); EXPANSION(y,26); EXPANSION(y,27);
						EXPANSION(y,28); EXPANSION(y,29); EXPANSION(y,30); EXPANSION(y,31);
						EXPANSION(y,32); EXPANSION(y,33); EXPANSION(y,34); EXPANSION(y,35);
						EXPANSION(y,36); EXPANSION(y,37); EXPANSION(y,38); EXPANSION(y,39);
						EXPANSION(y,40); EXPANSION(y,41); EXPANSION(y,42); EXPANSION(y,43);
						EXPANSION(y,44); EXPANSION(y,45); EXPANSION(y,46); EXPANSION(y,47);
						EXPANSION(y,48); EXPANSION(y,49); EXPANSION(y,50); EXPANSION(y,51);
						EXPANSION(y,52); EXPANSION(y,53); EXPANSION(y,54); EXPANSION(y,55);
						EXPANSION(y,56); EXPANSION(y,57); EXPANSION(y,58); EXPANSION(y,59);
						EXPANSION(y,60); EXPANSION(y,61); EXPANSION(y,62); EXPANSION(y,63);

						if (__XSHA1_INTERLACED__ > 0) { a[0] = hash[0]; }
						if (__XSHA1_INTERLACED__ > 1) { a[1] = hash[0]; }
						if (__XSHA1_INTERLACED__ > 0) { b[0] = hash[1]; }
						if (__XSHA1_INTERLACED__ > 1) { b[1] = hash[1]; }
						if (__XSHA1_INTERLACED__ > 0) { c[0] = hash[2]; }
						if (__XSHA1_INTERLACED__ > 1) { c[1] = hash[2]; }
						if (__XSHA1_INTERLACED__ > 0) { d[0] = hash[3]; }
						if (__XSHA1_INTERLACED__ > 1) { d[1] = hash[3]; }
						if (__XSHA1_INTERLACED__ > 0) { e[0] = hash[4]; }
						if (__XSHA1_INTERLACED__ > 1) { e[1] = hash[4]; }

						SHA1_II_REV(b,c,d,e,a,    y,79); SHA1_II_REV(c,d,e,a,b,    y,78); SHA1_II_REV(d,e,a,b,c,    y,77); SHA1_II_REV(e,a,b,c,d,    y,76); SHA1_II_REV(a,b,c,d,e,    y,75);
						SHA1_II_REV(b,c,d,e,a,    y,74); SHA1_II_REV(c,d,e,a,b,    y,73); SHA1_II_REV(d,e,a,b,c,    y,72); SHA1_II_REV(e,a,b,c,d,    y,71); SHA1_II_REV(a,b,c,d,e,    y,70);
						SHA1_II_REV(b,c,d,e,a,    y,69); SHA1_II_REV(c,d,e,a,b,    y,68); SHA1_II_REV(d,e,a,b,c,    y,67); SHA1_II_REV(e,a,b,c,d,    y,66); SHA1_II_REV(a,b,c,d,e,    y,65);
						SHA1_II_REV(b,c,d,e,a,    y,64); SHA1_II_REV(c,d,e,a,b,    y,63); SHA1_II_REV(d,e,a,b,c,    y,62); SHA1_II_REV(e,a,b,c,d,    y,61); SHA1_II_REV(a,b,c,d,e,    y,60);

						SHA1_HH_REV(b,c,d,e,a,    y,59); SHA1_HH_REV(c,d,e,a,b,    y,58); SHA1_HH_REV(d,e,a,b,c,    y,57); SHA1_HH_REV(e,a,b,c,d,    y,56); SHA1_HH_REV(a,b,c,d,e,    y,55);
						SHA1_HH_REV(b,c,d,e,a,    y,54); SHA1_HH_REV(c,d,e,a,b,    y,53); SHA1_HH_REV(d,e,a,b,c,    y,52); SHA1_HH_REV(e,a,b,c,d,    y,51); SHA1_HH_REV(a,b,c,d,e,    y,50);
						SHA1_HH_REV(b,c,d,e,a,    y,49); SHA1_HH_REV(c,d,e,a,b,    y,48); SHA1_HH_REV(d,e,a,b,c,    y,47); SHA1_HH_REV(e,a,b,c,d,    y,46); SHA1_HH_REV(a,b,c,d,e,    y,45);
						SHA1_HH_REV(b,c,d,e,a,    y,44); SHA1_HH_REV(c,d,e,a,b,    y,43); SHA1_HH_REV(d,e,a,b,c,    y,42); SHA1_HH_REV(e,a,b,c,d,    y,41); SHA1_HH_REV(a,b,c,d,e,    y,40);

						SHA1_GG_REV(b,c,d,e,a,    y,39); SHA1_GG_REV(c,d,e,a,b,    y,38); SHA1_GG_REV(d,e,a,b,c,    y,37); SHA1_GG_REV(e,a,b,c,d,    y,36); SHA1_GG_REV(a,b,c,d,e,    y,35);
						SHA1_GG_REV(b,c,d,e,a,    y,34); SHA1_GG_REV(c,d,e,a,b,    y,33); SHA1_GG_REV(d,e,a,b,c,    y,32); SHA1_GG_REV(e,a,b,c,d,    y,31); SHA1_GG_REV(a,b,c,d,e,    y,30);
						SHA1_GG_REV(b,c,d,e,a,    y,29); SHA1_GG_REV(c,d,e,a,b,    y,28); SHA1_GG_REV(d,e,a,b,c,    y,27); SHA1_GG_REV(e,a,b,c,d,    y,26); SHA1_GG_REV(a,b,c,d,e,    y,25);
						SHA1_GG_REV(b,c,d,e,a,    y,24); SHA1_GG_REV(c,d,e,a,b,    y,23); SHA1_GG_REV(d,e,a,b,c,    y,22); SHA1_GG_REV(e,a,b,c,d,    y,21); SHA1_GG_REV(a,b,c,d,e,    y,20);

						SHA1_FF_REV(b,c,d,e,a,    y,19); SHA1_FF_REV(c,d,e,a,b,    y,18); SHA1_FF_REV(d,e,a,b,c,    y,17);

						if (x0 == 5)
						{
							break;
						}

						SHA1_FF_REV(e,a,b,c,d,    y,16);

						SHA1_FF_REV(a,b,c,d,e,ZEROY,15);
						SHA1_FF_REV(b,c,d,e,a,ZEROY,14); SHA1_FF_REV(c,d,e,a,b,ZEROY,13); SHA1_FF_REV(d,e,a,b,c,ZEROY,12); SHA1_FF_REV(e,a,b,c,d,ZEROY,11); SHA1_FF_REV(a,b,c,d,e,ZEROY,10);
						SHA1_FF_REV(b,c,d,e,a,ZEROY, 9); SHA1_FF_REV(c,d,e,a,b,ZEROY, 8); SHA1_FF_REV(d,e,a,b,c,ZEROY, 7); SHA1_FF_REV(e,a,b,c,d,ZEROY, 6); SHA1_FF_REV(a,b,c,d,e,ZEROY, 5);

						if (__XSHA1_INTERLACED__ > 0)
						{
							//x[16] = ROT(e[0], 32 - 30) - (0xc3d2e1f0 + ((0xefcdab89 & 0x98badcfe) | ((~0xefcdab89) & 0x10325476)) + ROT(0x67452301, 5) + 0x5a827999);
							tmp = _mm_sub_epi32(ROT(e[0], 32 - 30), _mm_set1_epi32(0x9FB498B3));
							stmp = _mm_movemask_epi8(_mm_cmpeq_epi32(_mm_and_si128(tmp, _mm_set1_epi32(31)), y(0,0)));
							while (stmp != 0)
							{
								if (finishRev20(&tmp, a + 0, b + 0, c + 0, d + 0, e + 0, sx, x1, x2, x3, x4, stmp))
								{
									invalid = outputPassword(sx, onlyValid, onlyPrintable, h, fout);
									invalidCount += invalid;
									count++;
									if (invalid == 0 && noCollisions != 0)
									{
										return count - invalidCount;
									}
								}
							}
						}
						if (__XSHA1_INTERLACED__ > 1)
						{
							//x[16] = ROT(e[1], 32 - 30) - (0xc3d2e1f0 + ((0xefcdab89 & 0x98badcfe) | ((~0xefcdab89) & 0x10325476)) + ROT(0x67452301, 5) + 0x5a827999);
							tmp = _mm_sub_epi32(ROT(e[1], 32 - 30), _mm_set1_epi32(0x9FB498B3));
							stmp = _mm_movemask_epi8(_mm_cmpeq_epi32(_mm_and_si128(tmp, _mm_set1_epi32(31)), y(0,1)));
							while (stmp != 0)
							{
								if (finishRev20(&tmp, a + 1, b + 1, c + 1, d + 1, e + 1, sx, x1, x2, x3, x4 + 4, stmp))
								{
									invalid = outputPassword(sx, onlyValid, onlyPrintable, h, fout);
									invalidCount += invalid;
									count++;
									if (invalid == 0 && noCollisions != 0)
									{
										return count - invalidCount;
									}
								}
							}
						}
					}

					SHA1_FF_REV(e,a,b,c,d,ZEROY,16);
					if (__XSHA1_INTERLACED__ > 0) { t16a[0] = a[0]; }
					if (__XSHA1_INTERLACED__ > 1) { t16a[1] = a[1]; }
					if (__XSHA1_INTERLACED__ > 0) { t16b[0] = b[0]; }
					if (__XSHA1_INTERLACED__ > 1) { t16b[1] = b[1]; }
					if (__XSHA1_INTERLACED__ > 0) { t16c[0] = c[0]; }
					if (__XSHA1_INTERLACED__ > 1) { t16c[1] = c[1]; }
					if (__XSHA1_INTERLACED__ > 0) { t16d[0] = d[0]; }
					if (__XSHA1_INTERLACED__ > 1) { t16d[1] = d[1]; }
					if (__XSHA1_INTERLACED__ > 0) { t16e[0] = e[0]; }
					if (__XSHA1_INTERLACED__ > 1) { t16e[1] = e[1]; }
					// x[16] = (1 << 5) to (1 << 31)
#if __XSHA1_INTERLACED__ > 1
					for (; x0 < 32; x0++, y(16,1) = y(16,0) = _mm_slli_epi32(y(16,0), 1), y(0,1) = y(0,0) = _mm_xor_si128(y0 = _mm_add_epi32(y0, _mm_set1_epi32(1)), y(2,0)))
#elif __XSHA1_INTERLACED__ > 0
					for (; x0 < 32; x0++,           y(16,0) = _mm_slli_epi32(y(16,0), 1),          y(0,0) = _mm_xor_si128(y0 = _mm_add_epi32(y0, _mm_set1_epi32(1)), y(2,0)))
#endif
					{
						if (__XSHA1_INTERLACED__ > 0) { a[0] = t16a[0]; }
						if (__XSHA1_INTERLACED__ > 1) { a[1] = t16a[1]; }
						if (__XSHA1_INTERLACED__ > 0) { b[0] = t16b[0]; }
						if (__XSHA1_INTERLACED__ > 1) { b[1] = t16b[1]; }
						if (__XSHA1_INTERLACED__ > 0) { c[0] = t16c[0]; }
						if (__XSHA1_INTERLACED__ > 1) { c[1] = t16c[1]; }
						if (__XSHA1_INTERLACED__ > 0) { d[0] = _mm_sub_epi32(t16d[0], y(16,0)); }
						if (__XSHA1_INTERLACED__ > 1) { d[1] = _mm_sub_epi32(t16d[1], y(16,1)); }
						if (__XSHA1_INTERLACED__ > 0) { e[0] = t16e[0]; }
						if (__XSHA1_INTERLACED__ > 1) { e[1] = t16e[1]; }

						SHA1_FF_REV(a,b,c,d,e,ZEROY,15);
						SHA1_FF_REV(b,c,d,e,a,ZEROY,14); SHA1_FF_REV(c,d,e,a,b,ZEROY,13); SHA1_FF_REV(d,e,a,b,c,ZEROY,12); SHA1_FF_REV(e,a,b,c,d,ZEROY,11); SHA1_FF_REV(a,b,c,d,e,ZEROY,10);
						SHA1_FF_REV(b,c,d,e,a,ZEROY, 9); SHA1_FF_REV(c,d,e,a,b,ZEROY, 8); SHA1_FF_REV(d,e,a,b,c,ZEROY, 7); SHA1_FF_REV(e,a,b,c,d,ZEROY, 6); SHA1_FF_REV(a,b,c,d,e,ZEROY, 5);

						if (__XSHA1_INTERLACED__ > 0)
						{
							//x[16] = ROT(e[0], 32 - 30) - (0xc3d2e1f0 + ((0xefcdab89 & 0x98badcfe) | ((~0xefcdab89) & 0x10325476)) + ROT(0x67452301, 5) + 0x5a827999);
							tmp = _mm_sub_epi32(ROT(e[0], 32 - 30), _mm_set1_epi32(0x9FB498B3));
							stmp = _mm_movemask_epi8(_mm_cmpeq_epi32(_mm_and_si128(tmp, _mm_set1_epi32(31)), y(0,0)));
							while (stmp != 0)
							{
								if (finishRev20(&tmp, a + 0, b + 0, c + 0, d + 0, e + 0, sx, x1, x2, x3, x4, stmp))
								{
									invalid = outputPassword(sx, onlyValid, onlyPrintable, h, fout);
									invalidCount += invalid;
									count++;
									if (invalid == 0 && noCollisions != 0)
									{
										return count - invalidCount;
									}
								}
							}
						}
						if (__XSHA1_INTERLACED__ > 1)
						{
							//x[16] = ROT(e[1], 32 - 30) - (0xc3d2e1f0 + ((0xefcdab89 & 0x98badcfe) | ((~0xefcdab89) & 0x10325476)) + ROT(0x67452301, 5) + 0x5a827999);
							tmp = _mm_sub_epi32(ROT(e[1], 32 - 30), _mm_set1_epi32(0x9FB498B3));
							stmp = _mm_movemask_epi8(_mm_cmpeq_epi32(_mm_and_si128(tmp, _mm_set1_epi32(31)), y(0,1)));
							while (stmp != 0)
							{
								if (finishRev20(&tmp, a + 1, b + 1, c + 1, d + 1, e + 1, sx, x1, x2, x3, x4 + 4, stmp))
								{
									invalid = outputPassword(sx, onlyValid, onlyPrintable, h, fout);
									invalidCount += invalid;
									count++;
									if (invalid == 0 && noCollisions != 0)
									{
										return count - invalidCount;
									}
								}
							}
						}
					}
				}
			}
			if (__XSHA1_INTERLACED__ > 0) { y(1,0) = _mm_xor_si128(y(3,0), _mm_set1_epi32(5)); }
			if (__XSHA1_INTERLACED__ > 1) { y(1,1) = _mm_xor_si128(y(3,1), _mm_set1_epi32(5)); }
			if (__XSHA1_INTERLACED__ > 0) { y(2,0) = _mm_set1_epi32(0); }
			if (__XSHA1_INTERLACED__ > 1) { y(2,1) = _mm_set1_epi32(0); }
#if __XSHA1_INTERLACED__ > 1
			for (x2 = 0; x2 < 32; x2++, y(2,1) = y(2,0) = _mm_add_epi32(y(2,0), _mm_set1_epi32(1)))
#elif __XSHA1_INTERLACED__ > 0
			for (x2 = 0; x2 < 32; x2++,          y(2,0) = _mm_add_epi32(y(2,0), _mm_set1_epi32(1)))
#endif
			{
				y0 = _mm_set1_epi32(0);
				for (x0 = 0; /*x0 < 5*/; x0++, y0 = _mm_add_epi32(y0, _mm_set1_epi32(1)))
				{
					if (__XSHA1_INTERLACED__ > 0) { y( 0,0) = _mm_xor_si128(y(2,0), y0); }
					if (__XSHA1_INTERLACED__ > 1) { y( 0,1) = y( 0,0); }
					if (__XSHA1_INTERLACED__ > 0) { y(16,0) = POW_2_OF_X(y0); }
					if (__XSHA1_INTERLACED__ > 1) { y(16,1) = y(16,0); }
					if (__XSHA1_INTERLACED__ > 0) { y(17,0) = _mm_set1_epi32(1 << 5); }
					if (__XSHA1_INTERLACED__ > 1) { y(17,1) = _mm_set1_epi32(1 << 5); }
					if (__XSHA1_INTERLACED__ > 0) { y(18,0) = POW_2_OF_X(                            _mm_xor_si128(y( 2,0), y( 4,0))); }
					if (__XSHA1_INTERLACED__ > 1) { y(18,1) = POW_2_OF_X(                            _mm_xor_si128(y( 2,1), y( 4,1))); }
					if (__XSHA1_INTERLACED__ > 0) { y(19,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(y( 3,0),                     y(16,0)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(19,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(y( 3,1),                     y(16,1)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 0) { y(20,0) = POW_2_OF_X(y( 4,0)); }
					if (__XSHA1_INTERLACED__ > 1) { y(20,1) = POW_2_OF_X(y( 4,1)); }
					if (__XSHA1_INTERLACED__ > 0) { y(21,0) = POW_2_OF_X(_mm_and_si128(                                                         y(18,0) , _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(21,1) = POW_2_OF_X(_mm_and_si128(                                                         y(18,1) , _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 0) { y(22,0) = POW_2_OF_X(_mm_and_si128(                                                         y(19,0) , _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(22,1) = POW_2_OF_X(_mm_and_si128(                                                         y(19,1) , _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 0) { y(23,0) = POW_2_OF_X(_mm_and_si128(                                                         y(20,0) , _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(23,1) = POW_2_OF_X(_mm_and_si128(                                                         y(20,1) , _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 0) { y(24,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(16,0) , y(21,0)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(24,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(16,1) , y(21,1)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 0) { y(25,0) = POW_2_OF_X(_mm_and_si128(                                                         y(22,0) , _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(25,1) = POW_2_OF_X(_mm_and_si128(                                                         y(22,1) , _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 0) { y(26,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(18,0) , y(23,0)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(26,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(18,1) , y(23,1)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 0) { y(27,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(19,0) , y(24,0)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(27,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(19,1) , y(24,1)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 0) { y(28,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(20,0) , y(25,0)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(28,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(20,1) , y(25,1)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 0) { y(29,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(21,0) , y(26,0)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(29,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(21,1) , y(26,1)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 0) { y(30,0) = POW_2_OF_X(_mm_and_si128(_mm_xor_si128(_mm_xor_si128(         y(16,0) , y(22,0)), y(27,0)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(30,1) = POW_2_OF_X(_mm_and_si128(_mm_xor_si128(_mm_xor_si128(         y(16,1) , y(22,1)), y(27,1)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 0) { y(31,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(23,0) , y(28,0)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(31,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(23,1) , y(28,1)), _mm_set1_epi32(31))); }
					EXPANSION(y,16); EXPANSION(y,17); EXPANSION(y,18); EXPANSION(y,19);
					EXPANSION(y,20); EXPANSION(y,21); EXPANSION(y,22); EXPANSION(y,23);
					EXPANSION(y,24); EXPANSION(y,25); EXPANSION(y,26); EXPANSION(y,27);
					EXPANSION(y,28); EXPANSION(y,29); EXPANSION(y,30); EXPANSION(y,31);
					EXPANSION(y,32); EXPANSION(y,33); EXPANSION(y,34); EXPANSION(y,35);
					EXPANSION(y,36); EXPANSION(y,37); EXPANSION(y,38); EXPANSION(y,39);
					EXPANSION(y,40); EXPANSION(y,41); EXPANSION(y,42); EXPANSION(y,43);
					EXPANSION(y,44); EXPANSION(y,45); EXPANSION(y,46); EXPANSION(y,47);
					EXPANSION(y,48); EXPANSION(y,49); EXPANSION(y,50); EXPANSION(y,51);
					EXPANSION(y,52); EXPANSION(y,53); EXPANSION(y,54); EXPANSION(y,55);
					EXPANSION(y,56); EXPANSION(y,57); EXPANSION(y,58); EXPANSION(y,59);
					EXPANSION(y,60); EXPANSION(y,61); EXPANSION(y,62); EXPANSION(y,63);

					if (__XSHA1_INTERLACED__ > 0) { a[0] = hash[0]; }
					if (__XSHA1_INTERLACED__ > 1) { a[1] = hash[0]; }
					if (__XSHA1_INTERLACED__ > 0) { b[0] = hash[1]; }
					if (__XSHA1_INTERLACED__ > 1) { b[1] = hash[1]; }
					if (__XSHA1_INTERLACED__ > 0) { c[0] = hash[2]; }
					if (__XSHA1_INTERLACED__ > 1) { c[1] = hash[2]; }
					if (__XSHA1_INTERLACED__ > 0) { d[0] = hash[3]; }
					if (__XSHA1_INTERLACED__ > 1) { d[1] = hash[3]; }
					if (__XSHA1_INTERLACED__ > 0) { e[0] = hash[4]; }
					if (__XSHA1_INTERLACED__ > 1) { e[1] = hash[4]; }

					SHA1_II_REV(b,c,d,e,a,    y,79); SHA1_II_REV(c,d,e,a,b,    y,78); SHA1_II_REV(d,e,a,b,c,    y,77); SHA1_II_REV(e,a,b,c,d,    y,76); SHA1_II_REV(a,b,c,d,e,    y,75);
					SHA1_II_REV(b,c,d,e,a,    y,74); SHA1_II_REV(c,d,e,a,b,    y,73); SHA1_II_REV(d,e,a,b,c,    y,72); SHA1_II_REV(e,a,b,c,d,    y,71); SHA1_II_REV(a,b,c,d,e,    y,70);
					SHA1_II_REV(b,c,d,e,a,    y,69); SHA1_II_REV(c,d,e,a,b,    y,68); SHA1_II_REV(d,e,a,b,c,    y,67); SHA1_II_REV(e,a,b,c,d,    y,66); SHA1_II_REV(a,b,c,d,e,    y,65);
					SHA1_II_REV(b,c,d,e,a,    y,64); SHA1_II_REV(c,d,e,a,b,    y,63); SHA1_II_REV(d,e,a,b,c,    y,62); SHA1_II_REV(e,a,b,c,d,    y,61); SHA1_II_REV(a,b,c,d,e,    y,60);

					SHA1_HH_REV(b,c,d,e,a,    y,59); SHA1_HH_REV(c,d,e,a,b,    y,58); SHA1_HH_REV(d,e,a,b,c,    y,57); SHA1_HH_REV(e,a,b,c,d,    y,56); SHA1_HH_REV(a,b,c,d,e,    y,55);
					SHA1_HH_REV(b,c,d,e,a,    y,54); SHA1_HH_REV(c,d,e,a,b,    y,53); SHA1_HH_REV(d,e,a,b,c,    y,52); SHA1_HH_REV(e,a,b,c,d,    y,51); SHA1_HH_REV(a,b,c,d,e,    y,50);
					SHA1_HH_REV(b,c,d,e,a,    y,49); SHA1_HH_REV(c,d,e,a,b,    y,48); SHA1_HH_REV(d,e,a,b,c,    y,47); SHA1_HH_REV(e,a,b,c,d,    y,46); SHA1_HH_REV(a,b,c,d,e,    y,45);
					SHA1_HH_REV(b,c,d,e,a,    y,44); SHA1_HH_REV(c,d,e,a,b,    y,43); SHA1_HH_REV(d,e,a,b,c,    y,42); SHA1_HH_REV(e,a,b,c,d,    y,41); SHA1_HH_REV(a,b,c,d,e,    y,40);

					SHA1_GG_REV(b,c,d,e,a,    y,39); SHA1_GG_REV(c,d,e,a,b,    y,38); SHA1_GG_REV(d,e,a,b,c,    y,37); SHA1_GG_REV(e,a,b,c,d,    y,36); SHA1_GG_REV(a,b,c,d,e,    y,35);
					SHA1_GG_REV(b,c,d,e,a,    y,34); SHA1_GG_REV(c,d,e,a,b,    y,33); SHA1_GG_REV(d,e,a,b,c,    y,32); SHA1_GG_REV(e,a,b,c,d,    y,31); SHA1_GG_REV(a,b,c,d,e,    y,30);
					SHA1_GG_REV(b,c,d,e,a,    y,29); SHA1_GG_REV(c,d,e,a,b,    y,28); SHA1_GG_REV(d,e,a,b,c,    y,27); SHA1_GG_REV(e,a,b,c,d,    y,26); SHA1_GG_REV(a,b,c,d,e,    y,25);
					SHA1_GG_REV(b,c,d,e,a,    y,24); SHA1_GG_REV(c,d,e,a,b,    y,23); SHA1_GG_REV(d,e,a,b,c,    y,22); SHA1_GG_REV(e,a,b,c,d,    y,21); SHA1_GG_REV(a,b,c,d,e,    y,20);

					SHA1_FF_REV(b,c,d,e,a,    y,19); SHA1_FF_REV(c,d,e,a,b,    y,18);

					SHA1_FF_REV(d,e,a,b,c,ZEROY,17);
					if (__XSHA1_INTERLACED__ > 0) { t17a[0] = a[0]; }
					if (__XSHA1_INTERLACED__ > 1) { t17a[1] = a[1]; }
					if (__XSHA1_INTERLACED__ > 0) { t17b[0] = b[0]; }
					if (__XSHA1_INTERLACED__ > 1) { t17b[1] = b[1]; }
					if (__XSHA1_INTERLACED__ > 0) { t17c[0] = c[0]; }
					if (__XSHA1_INTERLACED__ > 1) { t17c[1] = c[1]; }
					if (__XSHA1_INTERLACED__ > 0) { t17d[0] = d[0]; }
					if (__XSHA1_INTERLACED__ > 1) { t17d[1] = d[1]; }
					if (__XSHA1_INTERLACED__ > 0) { t17e[0] = e[0]; }
					if (__XSHA1_INTERLACED__ > 1) { t17e[1] = e[1]; }

					if (x0 == 5)
					{
						break;
					}

					// x[16] = (1 << 5) to (1 << 31)
#if __XSHA1_INTERLACED__ > 1
					for (; x1 < 32; x1++, y(17,1) = y(17,0) = _mm_slli_epi32(y(17,0), 1), y(1,1) = y(1,0) = _mm_xor_si128(y1 = _mm_add_epi32(y1, _mm_set1_epi32(1)), y(3,0)))
#elif __XSHA1_INTERLACED__ > 0
					for (; x1 < 32; x1++,           y(17,0) = _mm_slli_epi32(y(17,0), 1),          y(1,0) = _mm_xor_si128(y1 = _mm_add_epi32(y1, _mm_set1_epi32(1)), y(3,0)))
#endif
					{
						if (__XSHA1_INTERLACED__ > 0) { a[0] = t17a[0]; }
						if (__XSHA1_INTERLACED__ > 1) { a[1] = t17a[1]; }
						if (__XSHA1_INTERLACED__ > 0) { b[0] = t17b[0]; }
						if (__XSHA1_INTERLACED__ > 1) { b[1] = t17b[1]; }
						if (__XSHA1_INTERLACED__ > 0) { c[0] = _mm_sub_epi32(t17c[0], y(17,0)); }
						if (__XSHA1_INTERLACED__ > 1) { c[1] = _mm_sub_epi32(t17c[1], y(17,1)); }
						if (__XSHA1_INTERLACED__ > 0) { d[0] = t17d[0]; }
						if (__XSHA1_INTERLACED__ > 1) { d[1] = t17d[1]; }
						if (__XSHA1_INTERLACED__ > 0) { e[0] = t17e[0]; }
						if (__XSHA1_INTERLACED__ > 1) { e[1] = t17e[1]; }

						SHA1_FF_REV(e,a,b,c,d,    y,16);

						SHA1_FF_REV(a,b,c,d,e,ZEROY,15);
						SHA1_FF_REV(b,c,d,e,a,ZEROY,14); SHA1_FF_REV(c,d,e,a,b,ZEROY,13); SHA1_FF_REV(d,e,a,b,c,ZEROY,12); SHA1_FF_REV(e,a,b,c,d,ZEROY,11); SHA1_FF_REV(a,b,c,d,e,ZEROY,10);
						SHA1_FF_REV(b,c,d,e,a,ZEROY, 9); SHA1_FF_REV(c,d,e,a,b,ZEROY, 8); SHA1_FF_REV(d,e,a,b,c,ZEROY, 7); SHA1_FF_REV(e,a,b,c,d,ZEROY, 6); SHA1_FF_REV(a,b,c,d,e,ZEROY, 5);

						if (__XSHA1_INTERLACED__ > 0)
						{
							//x[16] = ROT(e[0], 32 - 30) - (0xc3d2e1f0 + ((0xefcdab89 & 0x98badcfe) | ((~0xefcdab89) & 0x10325476)) + ROT(0x67452301, 5) + 0x5a827999);
							tmp = _mm_sub_epi32(ROT(e[0], 32 - 30), _mm_set1_epi32(0x9FB498B3));
							stmp = _mm_movemask_epi8(_mm_cmpeq_epi32(_mm_and_si128(tmp, _mm_set1_epi32(31)), y(0,0)));
							while (stmp != 0)
							{
								if (finishRev20(&tmp, a + 0, b + 0, c + 0, d + 0, e + 0, sx, x1, x2, x3, x4, stmp))
								{
									invalid = outputPassword(sx, onlyValid, onlyPrintable, h, fout);
									invalidCount += invalid;
									count++;
									if (invalid == 0 && noCollisions != 0)
									{
										return count - invalidCount;
									}
								}
							}
						}
						if (__XSHA1_INTERLACED__ > 1)
						{
							//x[16] = ROT(e[1], 32 - 30) - (0xc3d2e1f0 + ((0xefcdab89 & 0x98badcfe) | ((~0xefcdab89) & 0x10325476)) + ROT(0x67452301, 5) + 0x5a827999);
							tmp = _mm_sub_epi32(ROT(e[1], 32 - 30), _mm_set1_epi32(0x9FB498B3));
							stmp = _mm_movemask_epi8(_mm_cmpeq_epi32(_mm_and_si128(tmp, _mm_set1_epi32(31)), y(0,1)));
							while (stmp != 0)
							{
								if (finishRev20(&tmp, a + 1, b + 1, c + 1, d + 1, e + 1, sx, x1, x2, x3, x4 + 4, stmp))
								{
									invalid = outputPassword(sx, onlyValid, onlyPrintable, h, fout);
									invalidCount += invalid;
									count++;
									if (invalid == 0 && noCollisions != 0)
									{
										return count - invalidCount;
									}
								}
							}
						}
					}
					x1 = 5;
					y1 = _mm_set1_epi32(5);
					if (__XSHA1_INTERLACED__ > 0) { y(1,0) = _mm_xor_si128(y(3,0), _mm_set1_epi32(5)); }
					if (__XSHA1_INTERLACED__ > 1) { y(1,1) = _mm_xor_si128(y(3,1), _mm_set1_epi32(5)); }
				}

				// x[17] = (1 << 5) to (1 << 31)
#if __XSHA1_INTERLACED__ > 1
				for (; x1 < 32; x1++, y(17,1) = y(17,0) = _mm_slli_epi32(y(17,0), 1), y(1,1) = y(1,0) = _mm_xor_si128(y1 = _mm_add_epi32(y1, _mm_set1_epi32(1)), y(3,0)))
#elif __XSHA1_INTERLACED__ > 0
				for (; x1 < 32; x1++,           y(17,0) = _mm_slli_epi32(y(17,0), 1),          y(1,0) = _mm_xor_si128(y1 = _mm_add_epi32(y1, _mm_set1_epi32(1)), y(3,0)))
#endif
				{
					if (__XSHA1_INTERLACED__ > 0) { a[0] = t17a[0]; }
					if (__XSHA1_INTERLACED__ > 1) { a[1] = t17a[1]; }
					if (__XSHA1_INTERLACED__ > 0) { b[0] = t17b[0]; }
					if (__XSHA1_INTERLACED__ > 1) { b[1] = t17b[1]; }
					if (__XSHA1_INTERLACED__ > 0) { c[0] = _mm_sub_epi32(t17c[0], y(17,0)); }
					if (__XSHA1_INTERLACED__ > 1) { c[1] = _mm_sub_epi32(t17c[1], y(17,1)); }
					if (__XSHA1_INTERLACED__ > 0) { d[0] = t17d[0]; }
					if (__XSHA1_INTERLACED__ > 1) { d[1] = t17d[1]; }
					if (__XSHA1_INTERLACED__ > 0) { e[0] = t17e[0]; }
					if (__XSHA1_INTERLACED__ > 1) { e[1] = t17e[1]; }

					SHA1_FF_REV(e,a,b,c,d,ZEROY,16);
					if (__XSHA1_INTERLACED__ > 0) { t16a[0] = a[0]; }
					if (__XSHA1_INTERLACED__ > 1) { t16a[1] = a[1]; }
					if (__XSHA1_INTERLACED__ > 0) { t16b[0] = b[0]; }
					if (__XSHA1_INTERLACED__ > 1) { t16b[1] = b[1]; }
					if (__XSHA1_INTERLACED__ > 0) { t16c[0] = c[0]; }
					if (__XSHA1_INTERLACED__ > 1) { t16c[1] = c[1]; }
					if (__XSHA1_INTERLACED__ > 0) { t16d[0] = d[0]; }
					if (__XSHA1_INTERLACED__ > 1) { t16d[1] = d[1]; }
					if (__XSHA1_INTERLACED__ > 0) { t16e[0] = e[0]; }
					if (__XSHA1_INTERLACED__ > 1) { t16e[1] = e[1]; }
					// x[16] = (1 << 5) to (1 << 31)
#if __XSHA1_INTERLACED__ > 1
					for (; x0 < 32; x0++, y(16,1) = y(16,0) = _mm_slli_epi32(y(16,0), 1), y(0,1) = y(0,0) = _mm_xor_si128(y0 = _mm_add_epi32(y0, _mm_set1_epi32(1)), y(2,0)))
#elif __XSHA1_INTERLACED__ > 0
					for (; x0 < 32; x0++,           y(16,0) = _mm_slli_epi32(y(16,0), 1),          y(0,0) = _mm_xor_si128(y0 = _mm_add_epi32(y0, _mm_set1_epi32(1)), y(2,0)))
#endif
					{
						if (__XSHA1_INTERLACED__ > 0) { a[0] = t16a[0]; }
						if (__XSHA1_INTERLACED__ > 1) { a[1] = t16a[1]; }
						if (__XSHA1_INTERLACED__ > 0) { b[0] = t16b[0]; }
						if (__XSHA1_INTERLACED__ > 1) { b[1] = t16b[1]; }
						if (__XSHA1_INTERLACED__ > 0) { c[0] = t16c[0]; }
						if (__XSHA1_INTERLACED__ > 1) { c[1] = t16c[1]; }
						if (__XSHA1_INTERLACED__ > 0) { d[0] = _mm_sub_epi32(t16d[0], y(16,0)); }
						if (__XSHA1_INTERLACED__ > 1) { d[1] = _mm_sub_epi32(t16d[1], y(16,1)); }
						if (__XSHA1_INTERLACED__ > 0) { e[0] = t16e[0]; }
						if (__XSHA1_INTERLACED__ > 1) { e[1] = t16e[1]; }

						SHA1_FF_REV(a,b,c,d,e,ZEROY,15);
						SHA1_FF_REV(b,c,d,e,a,ZEROY,14); SHA1_FF_REV(c,d,e,a,b,ZEROY,13); SHA1_FF_REV(d,e,a,b,c,ZEROY,12); SHA1_FF_REV(e,a,b,c,d,ZEROY,11); SHA1_FF_REV(a,b,c,d,e,ZEROY,10);
						SHA1_FF_REV(b,c,d,e,a,ZEROY, 9); SHA1_FF_REV(c,d,e,a,b,ZEROY, 8); SHA1_FF_REV(d,e,a,b,c,ZEROY, 7); SHA1_FF_REV(e,a,b,c,d,ZEROY, 6); SHA1_FF_REV(a,b,c,d,e,ZEROY, 5);

						if (__XSHA1_INTERLACED__ > 0)
						{
							//x[16] = ROT(e[0], 32 - 30) - (0xc3d2e1f0 + ((0xefcdab89 & 0x98badcfe) | ((~0xefcdab89) & 0x10325476)) + ROT(0x67452301, 5) + 0x5a827999);
							tmp = _mm_sub_epi32(ROT(e[0], 32 - 30), _mm_set1_epi32(0x9FB498B3));
							stmp = _mm_movemask_epi8(_mm_cmpeq_epi32(_mm_and_si128(tmp, _mm_set1_epi32(31)), y(0,0)));
							while (stmp != 0)
							{
								if (finishRev20(&tmp, a + 0, b + 0, c + 0, d + 0, e + 0, sx, x1, x2, x3, x4, stmp))
								{
									invalid = outputPassword(sx, onlyValid, onlyPrintable, h, fout);
									invalidCount += invalid;
									count++;
									if (invalid == 0 && noCollisions != 0)
									{
										return count - invalidCount;
									}
								}
							}
						}
						if (__XSHA1_INTERLACED__ > 1)
						{
							//x[16] = ROT(e[1], 32 - 30) - (0xc3d2e1f0 + ((0xefcdab89 & 0x98badcfe) | ((~0xefcdab89) & 0x10325476)) + ROT(0x67452301, 5) + 0x5a827999);
							tmp = _mm_sub_epi32(ROT(e[1], 32 - 30), _mm_set1_epi32(0x9FB498B3));
							stmp = _mm_movemask_epi8(_mm_cmpeq_epi32(_mm_and_si128(tmp, _mm_set1_epi32(31)), y(0,1)));
							while (stmp != 0)
							{
								if (finishRev20(&tmp, a + 1, b + 1, c + 1, d + 1, e + 1, sx, x1, x2, x3, x4 + 4, stmp))
								{
									invalid = outputPassword(sx, onlyValid, onlyPrintable, h, fout);
									invalidCount += invalid;
									count++;
									if (invalid == 0 && noCollisions != 0)
									{
										return count - invalidCount;
									}
								}
							}
						}
					}
					x0 = 5;
					y0 = _mm_set1_epi32(5);
					if (__XSHA1_INTERLACED__ > 0) { y( 0,0) = _mm_xor_si128(y(2,0), _mm_set1_epi32(5)); }
					if (__XSHA1_INTERLACED__ > 1) { y( 0,1) = _mm_xor_si128(y(2,1), _mm_set1_epi32(5)); }
					if (__XSHA1_INTERLACED__ > 0) { y(16,0) = _mm_set1_epi32(1 << 5); }
					if (__XSHA1_INTERLACED__ > 1) { y(16,1) = _mm_set1_epi32(1 << 5); }
				}
				x1 = 5;
				y1 = _mm_set1_epi32(5);
				if (__XSHA1_INTERLACED__ > 0) { y(1,0) = _mm_xor_si128(y(3,0), _mm_set1_epi32(5)); }
				if (__XSHA1_INTERLACED__ > 1) { y(1,1) = _mm_xor_si128(y(3,1), _mm_set1_epi32(5)); }
			}
		}
	}
	return count - invalidCount;
}
#undef ROT

#ifdef XOP_CODE
#undef SHA1_FF_REV_
// ;( I want _mm_cmov_si128 and _mm_roti_epi32 damn you AMD why do you have all the good instructions... I might have to switch
#define SHA1_FF_REV_(a,b,c,d,e,x,i,j)  if (__XSHA1_INTERLACED__ > j) { e[j] = _mm_add_epi32(_mm_add_epi32(_mm_add_epi32(_mm_add_epi32(e[j], _mm_cmov_si128(c[j], d[j], b[j])), x(i,j)), LOAD_CONST_SHA1_FF), ROT(a[j], 5)); b[j] = ROT(b[j], 30); }
#define ROT(n,s) _mm_roti_epi32(n, s)
__inline__ uint32_t xsha1Reverse_16_XOP(const uint32_t *h, uint32_t &invalidCount, bool onlyValid, bool onlyPrintable, bool noCollisions, FILE *fout)
{
	__m128i a[__XSHA1_INTERLACED__], b[__XSHA1_INTERLACED__], c[__XSHA1_INTERLACED__], d[__XSHA1_INTERLACED__], e[__XSHA1_INTERLACED__], hash[5], y0, y1;
	__m128i t16a[__XSHA1_INTERLACED__], t16b[__XSHA1_INTERLACED__], t16c[__XSHA1_INTERLACED__], t16d[__XSHA1_INTERLACED__], t16e[__XSHA1_INTERLACED__];
	__m128i t17a[__XSHA1_INTERLACED__], t17b[__XSHA1_INTERLACED__], t17c[__XSHA1_INTERLACED__], t17d[__XSHA1_INTERLACED__], t17e[__XSHA1_INTERLACED__];
	__m128i x[80 * __XSHA1_INTERLACED__];
	uint32_t *sx = (uint32_t*) (x + 4 * __XSHA1_INTERLACED__), x0, x1, x2, x3, count = 0, stmp, invalid;

	invalidCount = 0;
	hash[0] = _mm_set1_epi32(h[0] - 0x67452301);
	hash[1] = _mm_set1_epi32(h[1] - 0xefcdab89);
	hash[2] = _mm_set1_epi32(h[2] - 0x98badcfe);
	hash[3] = _mm_set1_epi32(h[3] - 0x10325476);
	hash[4] = _mm_set1_epi32(h[4] - 0xc3d2e1f0);

	if (__XSHA1_INTERLACED__ > 0) { y(3,0) = _mm_set_epi32(3,2,1,0); }
	if (__XSHA1_INTERLACED__ > 1) { y(3,1) = _mm_set_epi32(7,6,5,4); }
#if __XSHA1_INTERLACED__ > 1
	for (x3 = 0; x3 < 32; x3 += 8, y(3,0) = _mm_add_epi32(y(3,0), _mm_set1_epi32(8)), y(3,1) = _mm_add_epi32(y(3,1), _mm_set1_epi32(8)))
#elif __XSHA1_INTERLACED__ > 0
	for (x3 = 0; x3 < 32; x3 += 4, y(3,0) = _mm_add_epi32(y(3,0), _mm_set1_epi32(4)))
#endif
	{
		y1 = _mm_set1_epi32(0);
		for (x1 = 0; x1 < 5; x1++, y1 = _mm_add_epi32(y1, _mm_set1_epi32(1)))
		{
			if (__XSHA1_INTERLACED__ > 0) { y(1,0) = _mm_xor_si128(y(3,0), y1); }
			if (__XSHA1_INTERLACED__ > 1) { y(1,1) = _mm_xor_si128(y(3,1), y1); }
			if (__XSHA1_INTERLACED__ > 0) { y(2,0) = _mm_set1_epi32(0); }
			if (__XSHA1_INTERLACED__ > 1) { y(2,1) = _mm_set1_epi32(0); }
#if __XSHA1_INTERLACED__ > 1
			for (x2 = 0; x2 < 32; x2++, y(2,1) = y(2,0) = _mm_add_epi32(y(2,0), _mm_set1_epi32(1)))
#elif __XSHA1_INTERLACED__ > 0
			for (x2 = 0; x2 < 32; x2++,          y(2,0) = _mm_add_epi32(y(2,0), _mm_set1_epi32(1)))
#endif
			{
				y0 = _mm_set1_epi32(0);
				for (x0 = 0; /*x0 < 5*/; x0++, y0 = _mm_add_epi32(y0, _mm_set1_epi32(1)))
				{
					if (__XSHA1_INTERLACED__ > 0) { y( 0,0) = _mm_xor_si128(y(2,0), y0); }
					if (__XSHA1_INTERLACED__ > 1) { y( 0,1) = y( 0,0); }
					if (__XSHA1_INTERLACED__ > 0) { y(16,0) = POW_2_OF_X(y0); }
					if (__XSHA1_INTERLACED__ > 1) { y(16,1) = y(16,0); }
					if (__XSHA1_INTERLACED__ > 0) { y(17,0) = POW_2_OF_X(y1); }
					if (__XSHA1_INTERLACED__ > 1) { y(17,1) = y(17,0); }
					if (__XSHA1_INTERLACED__ > 0) { y(18,0) = POW_2_OF_X(y(2,0)); }
					if (__XSHA1_INTERLACED__ > 1) { y(18,1) = y(18,0); }
					if (__XSHA1_INTERLACED__ > 0) { y(19,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(y( 3,0),                     y(16,0)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(19,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(y( 3,1),                     y(16,1)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 0) { y(20,0) = POW_2_OF_X(_mm_and_si128(                                                         y(17,0) , _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(20,1) = POW_2_OF_X(_mm_and_si128(                                                         y(17,1) , _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 0) { y(21,0) = POW_2_OF_X(_mm_and_si128(                                                         y(18,0) , _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(21,1) = POW_2_OF_X(_mm_and_si128(                                                         y(18,1) , _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 0) { y(22,0) = POW_2_OF_X(_mm_and_si128(                                                         y(19,0) , _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(22,1) = POW_2_OF_X(_mm_and_si128(                                                         y(19,1) , _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 0) { y(23,0) = POW_2_OF_X(_mm_and_si128(                                                         y(20,0) , _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(23,1) = POW_2_OF_X(_mm_and_si128(                                                         y(20,1) , _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 0) { y(24,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(16,0) , y(21,0)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(24,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(16,1) , y(21,1)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 0) { y(25,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(17,0) , y(22,0)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(25,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(17,1) , y(22,1)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 0) { y(26,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(18,0) , y(23,0)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(26,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(18,1) , y(23,1)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 0) { y(27,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(19,0) , y(24,0)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(27,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(19,1) , y(24,1)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 0) { y(28,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(20,0) , y(25,0)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(28,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(20,1) , y(25,1)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 0) { y(29,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(21,0) , y(26,0)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(29,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(21,1) , y(26,1)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 0) { y(30,0) = POW_2_OF_X(_mm_and_si128(_mm_xor_si128(_mm_xor_si128(         y(16,0) , y(22,0)), y(27,0)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(30,1) = POW_2_OF_X(_mm_and_si128(_mm_xor_si128(_mm_xor_si128(         y(16,1) , y(22,1)), y(27,1)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 0) { y(31,0) = POW_2_OF_X(_mm_and_si128(_mm_xor_si128(_mm_xor_si128(         y(17,0) , y(23,0)), y(28,0)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(31,1) = POW_2_OF_X(_mm_and_si128(_mm_xor_si128(_mm_xor_si128(         y(17,1) , y(23,1)), y(28,1)), _mm_set1_epi32(31))); }
					EXPANSION(y,16); EXPANSION(y,17); EXPANSION(y,18); EXPANSION(y,19);
					EXPANSION(y,20); EXPANSION(y,21); EXPANSION(y,22); EXPANSION(y,23);
					EXPANSION(y,24); EXPANSION(y,25); EXPANSION(y,26); EXPANSION(y,27);
					EXPANSION(y,28); EXPANSION(y,29); EXPANSION(y,30); EXPANSION(y,31);
					EXPANSION(y,32); EXPANSION(y,33); EXPANSION(y,34); EXPANSION(y,35);
					EXPANSION(y,36); EXPANSION(y,37); EXPANSION(y,38); EXPANSION(y,39);
					EXPANSION(y,40); EXPANSION(y,41); EXPANSION(y,42); EXPANSION(y,43);
					EXPANSION(y,44); EXPANSION(y,45); EXPANSION(y,46); EXPANSION(y,47);
					EXPANSION(y,48); EXPANSION(y,49); EXPANSION(y,50); EXPANSION(y,51);
					EXPANSION(y,52); EXPANSION(y,53); EXPANSION(y,54); EXPANSION(y,55);
					EXPANSION(y,56); EXPANSION(y,57); EXPANSION(y,58); EXPANSION(y,59);
					EXPANSION(y,60); EXPANSION(y,61); EXPANSION(y,62); EXPANSION(y,63);

					if (__XSHA1_INTERLACED__ > 0) { a[0] = hash[0]; }
					if (__XSHA1_INTERLACED__ > 1) { a[1] = hash[0]; }
					if (__XSHA1_INTERLACED__ > 0) { b[0] = hash[1]; }
					if (__XSHA1_INTERLACED__ > 1) { b[1] = hash[1]; }
					if (__XSHA1_INTERLACED__ > 0) { c[0] = hash[2]; }
					if (__XSHA1_INTERLACED__ > 1) { c[1] = hash[2]; }
					if (__XSHA1_INTERLACED__ > 0) { d[0] = hash[3]; }
					if (__XSHA1_INTERLACED__ > 1) { d[1] = hash[3]; }
					if (__XSHA1_INTERLACED__ > 0) { e[0] = hash[4]; }
					if (__XSHA1_INTERLACED__ > 1) { e[1] = hash[4]; }

					SHA1_II_REV(b,c,d,e,a,    y,79); SHA1_II_REV(c,d,e,a,b,    y,78); SHA1_II_REV(d,e,a,b,c,    y,77); SHA1_II_REV(e,a,b,c,d,    y,76); SHA1_II_REV(a,b,c,d,e,    y,75);
					SHA1_II_REV(b,c,d,e,a,    y,74); SHA1_II_REV(c,d,e,a,b,    y,73); SHA1_II_REV(d,e,a,b,c,    y,72); SHA1_II_REV(e,a,b,c,d,    y,71); SHA1_II_REV(a,b,c,d,e,    y,70);
					SHA1_II_REV(b,c,d,e,a,    y,69); SHA1_II_REV(c,d,e,a,b,    y,68); SHA1_II_REV(d,e,a,b,c,    y,67); SHA1_II_REV(e,a,b,c,d,    y,66); SHA1_II_REV(a,b,c,d,e,    y,65);
					SHA1_II_REV(b,c,d,e,a,    y,64); SHA1_II_REV(c,d,e,a,b,    y,63); SHA1_II_REV(d,e,a,b,c,    y,62); SHA1_II_REV(e,a,b,c,d,    y,61); SHA1_II_REV(a,b,c,d,e,    y,60);

					SHA1_HH_REV(b,c,d,e,a,    y,59); SHA1_HH_REV(c,d,e,a,b,    y,58); SHA1_HH_REV(d,e,a,b,c,    y,57); SHA1_HH_REV(e,a,b,c,d,    y,56); SHA1_HH_REV(a,b,c,d,e,    y,55);
					SHA1_HH_REV(b,c,d,e,a,    y,54); SHA1_HH_REV(c,d,e,a,b,    y,53); SHA1_HH_REV(d,e,a,b,c,    y,52); SHA1_HH_REV(e,a,b,c,d,    y,51); SHA1_HH_REV(a,b,c,d,e,    y,50);
					SHA1_HH_REV(b,c,d,e,a,    y,49); SHA1_HH_REV(c,d,e,a,b,    y,48); SHA1_HH_REV(d,e,a,b,c,    y,47); SHA1_HH_REV(e,a,b,c,d,    y,46); SHA1_HH_REV(a,b,c,d,e,    y,45);
					SHA1_HH_REV(b,c,d,e,a,    y,44); SHA1_HH_REV(c,d,e,a,b,    y,43); SHA1_HH_REV(d,e,a,b,c,    y,42); SHA1_HH_REV(e,a,b,c,d,    y,41); SHA1_HH_REV(a,b,c,d,e,    y,40);

					SHA1_GG_REV(b,c,d,e,a,    y,39); SHA1_GG_REV(c,d,e,a,b,    y,38); SHA1_GG_REV(d,e,a,b,c,    y,37); SHA1_GG_REV(e,a,b,c,d,    y,36); SHA1_GG_REV(a,b,c,d,e,    y,35);
					SHA1_GG_REV(b,c,d,e,a,    y,34); SHA1_GG_REV(c,d,e,a,b,    y,33); SHA1_GG_REV(d,e,a,b,c,    y,32); SHA1_GG_REV(e,a,b,c,d,    y,31); SHA1_GG_REV(a,b,c,d,e,    y,30);
					SHA1_GG_REV(b,c,d,e,a,    y,29); SHA1_GG_REV(c,d,e,a,b,    y,28); SHA1_GG_REV(d,e,a,b,c,    y,27); SHA1_GG_REV(e,a,b,c,d,    y,26); SHA1_GG_REV(a,b,c,d,e,    y,25);
					SHA1_GG_REV(b,c,d,e,a,    y,24); SHA1_GG_REV(c,d,e,a,b,    y,23); SHA1_GG_REV(d,e,a,b,c,    y,22); SHA1_GG_REV(e,a,b,c,d,    y,21); SHA1_GG_REV(a,b,c,d,e,    y,20);

					SHA1_FF_REV(b,c,d,e,a,    y,19); SHA1_FF_REV(c,d,e,a,b,    y,18); SHA1_FF_REV(d,e,a,b,c,    y,17);

					if (x0 == 5)
					{
						break;
					}

					SHA1_FF_REV(e,a,b,c,d,    y,16);

					SHA1_FF_REV(a,b,c,d,e,ZEROY,15);
					SHA1_FF_REV(b,c,d,e,a,ZEROY,14); SHA1_FF_REV(c,d,e,a,b,ZEROY,13); SHA1_FF_REV(d,e,a,b,c,ZEROY,12); SHA1_FF_REV(e,a,b,c,d,ZEROY,11); SHA1_FF_REV(a,b,c,d,e,ZEROY,10);
					SHA1_FF_REV(b,c,d,e,a,ZEROY, 9); SHA1_FF_REV(c,d,e,a,b,ZEROY, 8); SHA1_FF_REV(d,e,a,b,c,ZEROY, 7); SHA1_FF_REV(e,a,b,c,d,ZEROY, 6); SHA1_FF_REV(a,b,c,d,e,ZEROY, 5);
					SHA1_FF_REV(b,c,d,e,a,ZEROY, 4);

					if (__XSHA1_INTERLACED__ > 0)
					{
						stmp = _mm_movemask_epi8(_mm_cmpeq_epi32(a[0], _mm_set1_epi32(0x59d148c0)));
						while (stmp != 0)
						{
							if (finishRev16(b + 0, c + 0, d + 0, e + 0, sx, x0, x1, x2, x3, stmp))
							{
								invalid = outputPassword(sx, onlyValid, onlyPrintable, h, fout);
								invalidCount += invalid;
								count++;
								if (invalid == 0 && noCollisions != 0)
								{
									return count - invalidCount;
								}
							}
						}
					}
					if (__XSHA1_INTERLACED__ > 1)
					{
						stmp = _mm_movemask_epi8(_mm_cmpeq_epi32(a[1], _mm_set1_epi32(0x59d148c0)));
						while (stmp != 0)
						{
							if (finishRev16(b + 1, c + 1, d + 1, e + 1, sx, x0, x1, x2, x3 + 4, stmp))
							{
								invalid = outputPassword(sx, onlyValid, onlyPrintable, h, fout);
								invalidCount += invalid;
								count++;
								if (invalid == 0 && noCollisions != 0)
								{
									return count - invalidCount;
								}
							}
						}
					}
				}

				SHA1_FF_REV(e,a,b,c,d,ZEROY,16);
				if (__XSHA1_INTERLACED__ > 0) { t16a[0] = a[0]; }
				if (__XSHA1_INTERLACED__ > 1) { t16a[1] = a[1]; }
				if (__XSHA1_INTERLACED__ > 0) { t16b[0] = b[0]; }
				if (__XSHA1_INTERLACED__ > 1) { t16b[1] = b[1]; }
				if (__XSHA1_INTERLACED__ > 0) { t16c[0] = c[0]; }
				if (__XSHA1_INTERLACED__ > 1) { t16c[1] = c[1]; }
				if (__XSHA1_INTERLACED__ > 0) { t16d[0] = d[0]; }
				if (__XSHA1_INTERLACED__ > 1) { t16d[1] = d[1]; }
				if (__XSHA1_INTERLACED__ > 0) { t16e[0] = e[0]; }
				if (__XSHA1_INTERLACED__ > 1) { t16e[1] = e[1]; }
				// x[16] = (1 << 5) to (1 << 31)
#if __XSHA1_INTERLACED__ > 1
				for (; x0 < 32; x0++, y(16,1) = y(16,0) = _mm_slli_epi32(y(16,0), 1), y(0,1) = y(0,0) = _mm_xor_si128(y0 = _mm_add_epi32(y0, _mm_set1_epi32(1)), y(2,0)))
#elif __XSHA1_INTERLACED__ > 0
				for (; x0 < 32; x0++,           y(16,0) = _mm_slli_epi32(y(16,0), 1),          y(0,0) = _mm_xor_si128(y0 = _mm_add_epi32(y0, _mm_set1_epi32(1)), y(2,0)))
#endif
				{
					if (__XSHA1_INTERLACED__ > 0) { a[0] = t16a[0]; }
					if (__XSHA1_INTERLACED__ > 1) { a[1] = t16a[1]; }
					if (__XSHA1_INTERLACED__ > 0) { b[0] = t16b[0]; }
					if (__XSHA1_INTERLACED__ > 1) { b[1] = t16b[1]; }
					if (__XSHA1_INTERLACED__ > 0) { c[0] = t16c[0]; }
					if (__XSHA1_INTERLACED__ > 1) { c[1] = t16c[1]; }
					if (__XSHA1_INTERLACED__ > 0) { d[0] = _mm_sub_epi32(t16d[0], y(16,0)); }
					if (__XSHA1_INTERLACED__ > 1) { d[1] = _mm_sub_epi32(t16d[1], y(16,1)); }
					if (__XSHA1_INTERLACED__ > 0) { e[0] = t16e[0]; }
					if (__XSHA1_INTERLACED__ > 1) { e[1] = t16e[1]; }

					SHA1_FF_REV(a,b,c,d,e,ZEROY,15);
					SHA1_FF_REV(b,c,d,e,a,ZEROY,14); SHA1_FF_REV(c,d,e,a,b,ZEROY,13); SHA1_FF_REV(d,e,a,b,c,ZEROY,12); SHA1_FF_REV(e,a,b,c,d,ZEROY,11); SHA1_FF_REV(a,b,c,d,e,ZEROY,10);
					SHA1_FF_REV(b,c,d,e,a,ZEROY, 9); SHA1_FF_REV(c,d,e,a,b,ZEROY, 8); SHA1_FF_REV(d,e,a,b,c,ZEROY, 7); SHA1_FF_REV(e,a,b,c,d,ZEROY, 6); SHA1_FF_REV(a,b,c,d,e,ZEROY, 5);
					SHA1_FF_REV(b,c,d,e,a,ZEROY, 4);

					if (__XSHA1_INTERLACED__ > 0)
					{
						stmp = _mm_movemask_epi8(_mm_cmpeq_epi32(a[0], _mm_set1_epi32(0x59d148c0)));
						while (stmp != 0)
						{
							if (finishRev16(b + 0, c + 0, d + 0, e + 0, sx, x0, x1, x2, x3, stmp))
							{
								invalid = outputPassword(sx, onlyValid, onlyPrintable, h, fout);
								invalidCount += invalid;
								count++;
								if (invalid == 0 && noCollisions != 0)
								{
									return count - invalidCount;
								}
							}
						}
					}
					if (__XSHA1_INTERLACED__ > 1)
					{
						stmp = _mm_movemask_epi8(_mm_cmpeq_epi32(a[1], _mm_set1_epi32(0x59d148c0)));
						while (stmp != 0)
						{
							if (finishRev16(b + 1, c + 1, d + 1, e + 1, sx, x0, x1, x2, x3 + 4, stmp))
							{
								invalid = outputPassword(sx, onlyValid, onlyPrintable, h, fout);
								invalidCount += invalid;
								count++;
								if (invalid == 0 && noCollisions != 0)
								{
									return count - invalidCount;
								}
							}
						}
					}
				}
			}
		}
		if (__XSHA1_INTERLACED__ > 0) { y(1,0) = _mm_xor_si128(y(3,0), _mm_set1_epi32(5)); }
		if (__XSHA1_INTERLACED__ > 1) { y(1,1) = _mm_xor_si128(y(3,1), _mm_set1_epi32(5)); }
		if (__XSHA1_INTERLACED__ > 0) { y(2,0) = _mm_set1_epi32(0); }
		if (__XSHA1_INTERLACED__ > 1) { y(2,1) = _mm_set1_epi32(0); }
#if __XSHA1_INTERLACED__ > 1
		for (x2 = 0; x2 < 32; x2++, y(2,1) = y(2,0) = _mm_add_epi32(y(2,0), _mm_set1_epi32(1)))
#elif __XSHA1_INTERLACED__ > 0
		for (x2 = 0; x2 < 32; x2++,          y(2,0) = _mm_add_epi32(y(2,0), _mm_set1_epi32(1)))
#endif
		{
			y0 = _mm_set1_epi32(0);
			for (x0 = 0; /*x0 < 5*/; x0++, y0 = _mm_add_epi32(y0, _mm_set1_epi32(1)))
			{
				if (__XSHA1_INTERLACED__ > 0) { y( 0,0) = _mm_xor_si128(y(2,0), y0); }
				if (__XSHA1_INTERLACED__ > 1) { y( 0,1) = y( 0,0); }
				if (__XSHA1_INTERLACED__ > 0) { y(16,0) = POW_2_OF_X(y0); }
				if (__XSHA1_INTERLACED__ > 1) { y(16,1) = y(16,0); }
				if (__XSHA1_INTERLACED__ > 0) { y(17,0) = _mm_set1_epi32(1 << 5); }
				if (__XSHA1_INTERLACED__ > 1) { y(17,1) = _mm_set1_epi32(1 << 5); }
				if (__XSHA1_INTERLACED__ > 0) { y(18,0) = POW_2_OF_X(y(2,0)); }
				if (__XSHA1_INTERLACED__ > 1) { y(18,1) = y(18,0); }
				if (__XSHA1_INTERLACED__ > 0) { y(19,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(y( 3,0),                     y(16,0)), _mm_set1_epi32(31))); }
				if (__XSHA1_INTERLACED__ > 1) { y(19,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(y( 3,1),                     y(16,1)), _mm_set1_epi32(31))); }
				if (__XSHA1_INTERLACED__ > 0) { y(20,0) = _mm_set1_epi32(1); }
				if (__XSHA1_INTERLACED__ > 1) { y(20,1) = _mm_set1_epi32(1); }
				if (__XSHA1_INTERLACED__ > 0) { y(21,0) = POW_2_OF_X(_mm_and_si128(                                                         y(18,0) , _mm_set1_epi32(31))); }
				if (__XSHA1_INTERLACED__ > 1) { y(21,1) = POW_2_OF_X(_mm_and_si128(                                                         y(18,1) , _mm_set1_epi32(31))); }
				if (__XSHA1_INTERLACED__ > 0) { y(22,0) = POW_2_OF_X(_mm_and_si128(                                                         y(19,0) , _mm_set1_epi32(31))); }
				if (__XSHA1_INTERLACED__ > 1) { y(22,1) = POW_2_OF_X(_mm_and_si128(                                                         y(19,1) , _mm_set1_epi32(31))); }
				if (__XSHA1_INTERLACED__ > 0) { y(23,0) = POW_2_OF_X(_mm_and_si128(                                                         y(20,0) , _mm_set1_epi32(31))); }
				if (__XSHA1_INTERLACED__ > 1) { y(23,1) = POW_2_OF_X(_mm_and_si128(                                                         y(20,1) , _mm_set1_epi32(31))); }
				if (__XSHA1_INTERLACED__ > 0) { y(24,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(16,0) , y(21,0)), _mm_set1_epi32(31))); }
				if (__XSHA1_INTERLACED__ > 1) { y(24,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(16,1) , y(21,1)), _mm_set1_epi32(31))); }
				if (__XSHA1_INTERLACED__ > 0) { y(25,0) = POW_2_OF_X(_mm_and_si128(                                                         y(22,0) , _mm_set1_epi32(31))); }
				if (__XSHA1_INTERLACED__ > 1) { y(25,1) = POW_2_OF_X(_mm_and_si128(                                                         y(22,1) , _mm_set1_epi32(31))); }
				if (__XSHA1_INTERLACED__ > 0) { y(26,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(18,0) , y(23,0)), _mm_set1_epi32(31))); }
				if (__XSHA1_INTERLACED__ > 1) { y(26,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(18,1) , y(23,1)), _mm_set1_epi32(31))); }
				if (__XSHA1_INTERLACED__ > 0) { y(27,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(19,0) , y(24,0)), _mm_set1_epi32(31))); }
				if (__XSHA1_INTERLACED__ > 1) { y(27,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(19,1) , y(24,1)), _mm_set1_epi32(31))); }
				if (__XSHA1_INTERLACED__ > 0) { y(28,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(20,0) , y(25,0)), _mm_set1_epi32(31))); }
				if (__XSHA1_INTERLACED__ > 1) { y(28,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(20,1) , y(25,1)), _mm_set1_epi32(31))); }
				if (__XSHA1_INTERLACED__ > 0) { y(29,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(21,0) , y(26,0)), _mm_set1_epi32(31))); }
				if (__XSHA1_INTERLACED__ > 1) { y(29,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(21,1) , y(26,1)), _mm_set1_epi32(31))); }
				if (__XSHA1_INTERLACED__ > 0) { y(30,0) = POW_2_OF_X(_mm_and_si128(_mm_xor_si128(_mm_xor_si128(         y(16,0) , y(22,0)), y(27,0)), _mm_set1_epi32(31))); }
				if (__XSHA1_INTERLACED__ > 1) { y(30,1) = POW_2_OF_X(_mm_and_si128(_mm_xor_si128(_mm_xor_si128(         y(16,1) , y(22,1)), y(27,1)), _mm_set1_epi32(31))); }
				if (__XSHA1_INTERLACED__ > 0) { y(31,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(23,0) , y(28,0)), _mm_set1_epi32(31))); }
				if (__XSHA1_INTERLACED__ > 1) { y(31,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(23,1) , y(28,1)), _mm_set1_epi32(31))); }
				EXPANSION(y,16); EXPANSION(y,17); EXPANSION(y,18); EXPANSION(y,19);
				EXPANSION(y,20); EXPANSION(y,21); EXPANSION(y,22); EXPANSION(y,23);
				EXPANSION(y,24); EXPANSION(y,25); EXPANSION(y,26); EXPANSION(y,27);
				EXPANSION(y,28); EXPANSION(y,29); EXPANSION(y,30); EXPANSION(y,31);
				EXPANSION(y,32); EXPANSION(y,33); EXPANSION(y,34); EXPANSION(y,35);
				EXPANSION(y,36); EXPANSION(y,37); EXPANSION(y,38); EXPANSION(y,39);
				EXPANSION(y,40); EXPANSION(y,41); EXPANSION(y,42); EXPANSION(y,43);
				EXPANSION(y,44); EXPANSION(y,45); EXPANSION(y,46); EXPANSION(y,47);
				EXPANSION(y,48); EXPANSION(y,49); EXPANSION(y,50); EXPANSION(y,51);
				EXPANSION(y,52); EXPANSION(y,53); EXPANSION(y,54); EXPANSION(y,55);
				EXPANSION(y,56); EXPANSION(y,57); EXPANSION(y,58); EXPANSION(y,59);
				EXPANSION(y,60); EXPANSION(y,61); EXPANSION(y,62); EXPANSION(y,63);

				if (__XSHA1_INTERLACED__ > 0) { a[0] = hash[0]; }
				if (__XSHA1_INTERLACED__ > 1) { a[1] = hash[0]; }
				if (__XSHA1_INTERLACED__ > 0) { b[0] = hash[1]; }
				if (__XSHA1_INTERLACED__ > 1) { b[1] = hash[1]; }
				if (__XSHA1_INTERLACED__ > 0) { c[0] = hash[2]; }
				if (__XSHA1_INTERLACED__ > 1) { c[1] = hash[2]; }
				if (__XSHA1_INTERLACED__ > 0) { d[0] = hash[3]; }
				if (__XSHA1_INTERLACED__ > 1) { d[1] = hash[3]; }
				if (__XSHA1_INTERLACED__ > 0) { e[0] = hash[4]; }
				if (__XSHA1_INTERLACED__ > 1) { e[1] = hash[4]; }

				SHA1_II_REV(b,c,d,e,a,    y,79); SHA1_II_REV(c,d,e,a,b,    y,78); SHA1_II_REV(d,e,a,b,c,    y,77); SHA1_II_REV(e,a,b,c,d,    y,76); SHA1_II_REV(a,b,c,d,e,    y,75);
				SHA1_II_REV(b,c,d,e,a,    y,74); SHA1_II_REV(c,d,e,a,b,    y,73); SHA1_II_REV(d,e,a,b,c,    y,72); SHA1_II_REV(e,a,b,c,d,    y,71); SHA1_II_REV(a,b,c,d,e,    y,70);
				SHA1_II_REV(b,c,d,e,a,    y,69); SHA1_II_REV(c,d,e,a,b,    y,68); SHA1_II_REV(d,e,a,b,c,    y,67); SHA1_II_REV(e,a,b,c,d,    y,66); SHA1_II_REV(a,b,c,d,e,    y,65);
				SHA1_II_REV(b,c,d,e,a,    y,64); SHA1_II_REV(c,d,e,a,b,    y,63); SHA1_II_REV(d,e,a,b,c,    y,62); SHA1_II_REV(e,a,b,c,d,    y,61); SHA1_II_REV(a,b,c,d,e,    y,60);

				SHA1_HH_REV(b,c,d,e,a,    y,59); SHA1_HH_REV(c,d,e,a,b,    y,58); SHA1_HH_REV(d,e,a,b,c,    y,57); SHA1_HH_REV(e,a,b,c,d,    y,56); SHA1_HH_REV(a,b,c,d,e,    y,55);
				SHA1_HH_REV(b,c,d,e,a,    y,54); SHA1_HH_REV(c,d,e,a,b,    y,53); SHA1_HH_REV(d,e,a,b,c,    y,52); SHA1_HH_REV(e,a,b,c,d,    y,51); SHA1_HH_REV(a,b,c,d,e,    y,50);
				SHA1_HH_REV(b,c,d,e,a,    y,49); SHA1_HH_REV(c,d,e,a,b,    y,48); SHA1_HH_REV(d,e,a,b,c,    y,47); SHA1_HH_REV(e,a,b,c,d,    y,46); SHA1_HH_REV(a,b,c,d,e,    y,45);
				SHA1_HH_REV(b,c,d,e,a,    y,44); SHA1_HH_REV(c,d,e,a,b,    y,43); SHA1_HH_REV(d,e,a,b,c,    y,42); SHA1_HH_REV(e,a,b,c,d,    y,41); SHA1_HH_REV(a,b,c,d,e,    y,40);

				SHA1_GG_REV(b,c,d,e,a,    y,39); SHA1_GG_REV(c,d,e,a,b,    y,38); SHA1_GG_REV(d,e,a,b,c,    y,37); SHA1_GG_REV(e,a,b,c,d,    y,36); SHA1_GG_REV(a,b,c,d,e,    y,35);
				SHA1_GG_REV(b,c,d,e,a,    y,34); SHA1_GG_REV(c,d,e,a,b,    y,33); SHA1_GG_REV(d,e,a,b,c,    y,32); SHA1_GG_REV(e,a,b,c,d,    y,31); SHA1_GG_REV(a,b,c,d,e,    y,30);
				SHA1_GG_REV(b,c,d,e,a,    y,29); SHA1_GG_REV(c,d,e,a,b,    y,28); SHA1_GG_REV(d,e,a,b,c,    y,27); SHA1_GG_REV(e,a,b,c,d,    y,26); SHA1_GG_REV(a,b,c,d,e,    y,25);
				SHA1_GG_REV(b,c,d,e,a,    y,24); SHA1_GG_REV(c,d,e,a,b,    y,23); SHA1_GG_REV(d,e,a,b,c,    y,22); SHA1_GG_REV(e,a,b,c,d,    y,21); SHA1_GG_REV(a,b,c,d,e,    y,20);

				SHA1_FF_REV(b,c,d,e,a,    y,19); SHA1_FF_REV(c,d,e,a,b,    y,18);

				SHA1_FF_REV(d,e,a,b,c,ZEROY,17);
				if (__XSHA1_INTERLACED__ > 0) { t17a[0] = a[0]; }
				if (__XSHA1_INTERLACED__ > 1) { t17a[1] = a[1]; }
				if (__XSHA1_INTERLACED__ > 0) { t17b[0] = b[0]; }
				if (__XSHA1_INTERLACED__ > 1) { t17b[1] = b[1]; }
				if (__XSHA1_INTERLACED__ > 0) { t17c[0] = c[0]; }
				if (__XSHA1_INTERLACED__ > 1) { t17c[1] = c[1]; }
				if (__XSHA1_INTERLACED__ > 0) { t17d[0] = d[0]; }
				if (__XSHA1_INTERLACED__ > 1) { t17d[1] = d[1]; }
				if (__XSHA1_INTERLACED__ > 0) { t17e[0] = e[0]; }
				if (__XSHA1_INTERLACED__ > 1) { t17e[1] = e[1]; }

				if (x0 == 5)
				{
					break;
				}

				// x[16] = (1 << 5) to (1 << 31)
#if __XSHA1_INTERLACED__ > 1
				for (; x1 < 32; x1++, y(17,1) = y(17,0) = _mm_slli_epi32(y(17,0), 1), y(1,1) = y(1,0) = _mm_xor_si128(y1 = _mm_add_epi32(y1, _mm_set1_epi32(1)), y(3,0)))
#elif __XSHA1_INTERLACED__ > 0
				for (; x1 < 32; x1++,           y(17,0) = _mm_slli_epi32(y(17,0), 1),          y(1,0) = _mm_xor_si128(y1 = _mm_add_epi32(y1, _mm_set1_epi32(1)), y(3,0)))
#endif
				{
					if (__XSHA1_INTERLACED__ > 0) { a[0] = t17a[0]; }
					if (__XSHA1_INTERLACED__ > 1) { a[1] = t17a[1]; }
					if (__XSHA1_INTERLACED__ > 0) { b[0] = t17b[0]; }
					if (__XSHA1_INTERLACED__ > 1) { b[1] = t17b[1]; }
					if (__XSHA1_INTERLACED__ > 0) { c[0] = _mm_sub_epi32(t17c[0], y(17,0)); }
					if (__XSHA1_INTERLACED__ > 1) { c[1] = _mm_sub_epi32(t17c[1], y(17,1)); }
					if (__XSHA1_INTERLACED__ > 0) { d[0] = t17d[0]; }
					if (__XSHA1_INTERLACED__ > 1) { d[1] = t17d[1]; }
					if (__XSHA1_INTERLACED__ > 0) { e[0] = t17e[0]; }
					if (__XSHA1_INTERLACED__ > 1) { e[1] = t17e[1]; }

					SHA1_FF_REV(e,a,b,c,d,    y,16);

					SHA1_FF_REV(a,b,c,d,e,ZEROY,15);
					SHA1_FF_REV(b,c,d,e,a,ZEROY,14); SHA1_FF_REV(c,d,e,a,b,ZEROY,13); SHA1_FF_REV(d,e,a,b,c,ZEROY,12); SHA1_FF_REV(e,a,b,c,d,ZEROY,11); SHA1_FF_REV(a,b,c,d,e,ZEROY,10);
					SHA1_FF_REV(b,c,d,e,a,ZEROY, 9); SHA1_FF_REV(c,d,e,a,b,ZEROY, 8); SHA1_FF_REV(d,e,a,b,c,ZEROY, 7); SHA1_FF_REV(e,a,b,c,d,ZEROY, 6); SHA1_FF_REV(a,b,c,d,e,ZEROY, 5);
					SHA1_FF_REV(b,c,d,e,a,ZEROY, 4);

					if (__XSHA1_INTERLACED__ > 0)
					{
						stmp = _mm_movemask_epi8(_mm_cmpeq_epi32(a[0], _mm_set1_epi32(0x59d148c0)));
						while (stmp != 0)
						{
							if (finishRev16(b + 0, c + 0, d + 0, e + 0, sx, x0, x1, x2, x3, stmp))
							{
								invalid = outputPassword(sx, onlyValid, onlyPrintable, h, fout);
								invalidCount += invalid;
								count++;
								if (invalid == 0 && noCollisions != 0)
								{
									return count - invalidCount;
								}
							}
						}
					}
					if (__XSHA1_INTERLACED__ > 1)
					{
						stmp = _mm_movemask_epi8(_mm_cmpeq_epi32(a[1], _mm_set1_epi32(0x59d148c0)));
						while (stmp != 0)
						{
							if (finishRev16(b + 1, c + 1, d + 1, e + 1, sx, x0, x1, x2, x3 + 4, stmp))
							{
								invalid = outputPassword(sx, onlyValid, onlyPrintable, h, fout);
								invalidCount += invalid;
								count++;
								if (invalid == 0 && noCollisions != 0)
								{
									return count - invalidCount;
								}
							}
						}
					}
				}
				x1 = 5;
				y1 = _mm_set1_epi32(5);
				if (__XSHA1_INTERLACED__ > 0) { y(1,0) = _mm_xor_si128(y(3,0), _mm_set1_epi32(5)); }
				if (__XSHA1_INTERLACED__ > 1) { y(1,1) = _mm_xor_si128(y(3,1), _mm_set1_epi32(5)); }
			}

			// x[17] = (1 << 5) to (1 << 31)
#if __XSHA1_INTERLACED__ > 1
			for (; x1 < 32; x1++, y(17,1) = y(17,0) = _mm_slli_epi32(y(17,0), 1), y(1,1) = y(1,0) = _mm_xor_si128(y1 = _mm_add_epi32(y1, _mm_set1_epi32(1)), y(3,0)))
#elif __XSHA1_INTERLACED__ > 0
			for (; x1 < 32; x1++,           y(17,0) = _mm_slli_epi32(y(17,0), 1),          y(1,0) = _mm_xor_si128(y1 = _mm_add_epi32(y1, _mm_set1_epi32(1)), y(3,0)))
#endif
			{
				if (__XSHA1_INTERLACED__ > 0) { a[0] = t17a[0]; }
				if (__XSHA1_INTERLACED__ > 1) { a[1] = t17a[1]; }
				if (__XSHA1_INTERLACED__ > 0) { b[0] = t17b[0]; }
				if (__XSHA1_INTERLACED__ > 1) { b[1] = t17b[1]; }
				if (__XSHA1_INTERLACED__ > 0) { c[0] = _mm_sub_epi32(t17c[0], y(17,0)); }
				if (__XSHA1_INTERLACED__ > 1) { c[1] = _mm_sub_epi32(t17c[1], y(17,1)); }
				if (__XSHA1_INTERLACED__ > 0) { d[0] = t17d[0]; }
				if (__XSHA1_INTERLACED__ > 1) { d[1] = t17d[1]; }
				if (__XSHA1_INTERLACED__ > 0) { e[0] = t17e[0]; }
				if (__XSHA1_INTERLACED__ > 1) { e[1] = t17e[1]; }

				SHA1_FF_REV(e,a,b,c,d,ZEROY,16);
				if (__XSHA1_INTERLACED__ > 0) { t16a[0] = a[0]; }
				if (__XSHA1_INTERLACED__ > 1) { t16a[1] = a[1]; }
				if (__XSHA1_INTERLACED__ > 0) { t16b[0] = b[0]; }
				if (__XSHA1_INTERLACED__ > 1) { t16b[1] = b[1]; }
				if (__XSHA1_INTERLACED__ > 0) { t16c[0] = c[0]; }
				if (__XSHA1_INTERLACED__ > 1) { t16c[1] = c[1]; }
				if (__XSHA1_INTERLACED__ > 0) { t16d[0] = d[0]; }
				if (__XSHA1_INTERLACED__ > 1) { t16d[1] = d[1]; }
				if (__XSHA1_INTERLACED__ > 0) { t16e[0] = e[0]; }
				if (__XSHA1_INTERLACED__ > 1) { t16e[1] = e[1]; }
				// x[16] = (1 << 5) to (1 << 31)
#if __XSHA1_INTERLACED__ > 1
				for (; x0 < 32; x0++, y(16,1) = y(16,0) = _mm_slli_epi32(y(16,0), 1), y(0,1) = y(0,0) = _mm_xor_si128(y0 = _mm_add_epi32(y0, _mm_set1_epi32(1)), y(2,0)))
#elif __XSHA1_INTERLACED__ > 0
				for (; x0 < 32; x0++,           y(16,0) = _mm_slli_epi32(y(16,0), 1),          y(0,0) = _mm_xor_si128(y0 = _mm_add_epi32(y0, _mm_set1_epi32(1)), y(2,0)))
#endif
				{
					if (__XSHA1_INTERLACED__ > 0) { a[0] = t16a[0]; }
					if (__XSHA1_INTERLACED__ > 1) { a[1] = t16a[1]; }
					if (__XSHA1_INTERLACED__ > 0) { b[0] = t16b[0]; }
					if (__XSHA1_INTERLACED__ > 1) { b[1] = t16b[1]; }
					if (__XSHA1_INTERLACED__ > 0) { c[0] = t16c[0]; }
					if (__XSHA1_INTERLACED__ > 1) { c[1] = t16c[1]; }
					if (__XSHA1_INTERLACED__ > 0) { d[0] = _mm_sub_epi32(t16d[0], y(16,0)); }
					if (__XSHA1_INTERLACED__ > 1) { d[1] = _mm_sub_epi32(t16d[1], y(16,1)); }
					if (__XSHA1_INTERLACED__ > 0) { e[0] = t16e[0]; }
					if (__XSHA1_INTERLACED__ > 1) { e[1] = t16e[1]; }

					SHA1_FF_REV(a,b,c,d,e,ZEROY,15);
					SHA1_FF_REV(b,c,d,e,a,ZEROY,14); SHA1_FF_REV(c,d,e,a,b,ZEROY,13); SHA1_FF_REV(d,e,a,b,c,ZEROY,12); SHA1_FF_REV(e,a,b,c,d,ZEROY,11); SHA1_FF_REV(a,b,c,d,e,ZEROY,10);
					SHA1_FF_REV(b,c,d,e,a,ZEROY, 9); SHA1_FF_REV(c,d,e,a,b,ZEROY, 8); SHA1_FF_REV(d,e,a,b,c,ZEROY, 7); SHA1_FF_REV(e,a,b,c,d,ZEROY, 6); SHA1_FF_REV(a,b,c,d,e,ZEROY, 5);
					SHA1_FF_REV(b,c,d,e,a,ZEROY, 4);

					if (__XSHA1_INTERLACED__ > 0)
					{
						stmp = _mm_movemask_epi8(_mm_cmpeq_epi32(a[0], _mm_set1_epi32(0x59d148c0)));
						while (stmp != 0)
						{
							if (finishRev16(b + 0, c + 0, d + 0, e + 0, sx, x0, x1, x2, x3, stmp))
							{
								invalid = outputPassword(sx, onlyValid, onlyPrintable, h, fout);
								invalidCount += invalid;
								count++;
								if (invalid == 0 && noCollisions != 0)
								{
									return count - invalidCount;
								}
							}
						}
					}
					if (__XSHA1_INTERLACED__ > 1)
					{
						stmp = _mm_movemask_epi8(_mm_cmpeq_epi32(a[1], _mm_set1_epi32(0x59d148c0)));
						while (stmp != 0)
						{
							if (finishRev16(b + 1, c + 1, d + 1, e + 1, sx, x0, x1, x2, x3 + 4, stmp))
							{
								invalid = outputPassword(sx, onlyValid, onlyPrintable, h, fout);
								invalidCount += invalid;
								count++;
								if (invalid == 0 && noCollisions != 0)
								{
									return count - invalidCount;
								}
							}
						}
					}
				}
				x0 = 5;
				y0 = _mm_set1_epi32(5);
				if (__XSHA1_INTERLACED__ > 0) { y( 0,0) = _mm_xor_si128(y(2,0), _mm_set1_epi32(5)); }
				if (__XSHA1_INTERLACED__ > 1) { y( 0,1) = _mm_xor_si128(y(2,1), _mm_set1_epi32(5)); }
				if (__XSHA1_INTERLACED__ > 0) { y(16,0) = _mm_set1_epi32(1 << 5); }
				if (__XSHA1_INTERLACED__ > 1) { y(16,1) = _mm_set1_epi32(1 << 5); }
			}
			x1 = 5;
			y1 = _mm_set1_epi32(5);
			if (__XSHA1_INTERLACED__ > 0) { y(1,0) = _mm_xor_si128(y(3,0), _mm_set1_epi32(5)); }
			if (__XSHA1_INTERLACED__ > 1) { y(1,1) = _mm_xor_si128(y(3,1), _mm_set1_epi32(5)); }
		}
	}
	return count - invalidCount;
}

__inline__ uint32_t xsha1Reverse_20_XOP(const uint32_t *h, uint32_t &invalidCount, bool onlyValid, bool onlyPrintable, bool noCollisions, FILE *fout)
{
	__m128i a[__XSHA1_INTERLACED__], b[__XSHA1_INTERLACED__], c[__XSHA1_INTERLACED__], d[__XSHA1_INTERLACED__], e[__XSHA1_INTERLACED__], tmp, hash[5], y0, y1;
	__m128i t16a[__XSHA1_INTERLACED__], t16b[__XSHA1_INTERLACED__], t16c[__XSHA1_INTERLACED__], t16d[__XSHA1_INTERLACED__], t16e[__XSHA1_INTERLACED__];
	__m128i t17a[__XSHA1_INTERLACED__], t17b[__XSHA1_INTERLACED__], t17c[__XSHA1_INTERLACED__], t17d[__XSHA1_INTERLACED__], t17e[__XSHA1_INTERLACED__];
	__m128i x[80 * __XSHA1_INTERLACED__];
	uint32_t *sx = (uint32_t*) (x + 5 * __XSHA1_INTERLACED__), x0, x1, x2, x3, x4, count = 0, stmp, invalid;

	invalidCount = 0;
	hash[0] = _mm_set1_epi32(h[0] - 0x67452301);
	hash[1] = _mm_set1_epi32(h[1] - 0xefcdab89);
	hash[2] = _mm_set1_epi32(h[2] - 0x98badcfe);
	hash[3] = _mm_set1_epi32(h[3] - 0x10325476);
	hash[4] = _mm_set1_epi32(h[4] - 0xc3d2e1f0);

	if (__XSHA1_INTERLACED__ > 0) { y(4,0) = _mm_set_epi32(3,2,1,0); }
	if (__XSHA1_INTERLACED__ > 1) { y(4,1) = _mm_set_epi32(7,6,5,4); }
#if __XSHA1_INTERLACED__ > 1
	for (x4 = 0; x4 < 32; x4 += 8, y(4,0) = _mm_add_epi32(y(4,0), _mm_set1_epi32(8)), y(4,1) = _mm_add_epi32(y(4,1), _mm_set1_epi32(8)))
#elif __XSHA1_INTERLACED__ > 0
	for (x4 = 0; x4 < 32; x4 += 4, y(4,0) = _mm_add_epi32(y(4,0), _mm_set1_epi32(4)))
#endif
	{
		if (__XSHA1_INTERLACED__ > 0) { y(3,0) = _mm_set1_epi32(0); }
		if (__XSHA1_INTERLACED__ > 1) { y(3,1) = _mm_set1_epi32(0); }
#if __XSHA1_INTERLACED__ > 1
		for (x3 = 0; x3 < 32; x3++, y(3,0) = _mm_add_epi32(y(3,0), _mm_set1_epi32(1)), y(3,1) = _mm_add_epi32(y(3,1), _mm_set1_epi32(1)))
#elif __XSHA1_INTERLACED__ > 0
		for (x3 = 0; x3 < 32; x3++, y(3,0) = _mm_add_epi32(y(3,0), _mm_set1_epi32(1)))
#endif
		{
			y1 = _mm_set1_epi32(0);
			for (x1 = 0; x1 < 5; x1++, y1 = _mm_add_epi32(y1, _mm_set1_epi32(1)))
			{
				if (__XSHA1_INTERLACED__ > 0) { y(1,0) = _mm_xor_si128(y(3,0), y1); }
				if (__XSHA1_INTERLACED__ > 1) { y(1,1) = _mm_xor_si128(y(3,1), y1); }
				if (__XSHA1_INTERLACED__ > 0) { y(2,0) = _mm_set1_epi32(0); }
				if (__XSHA1_INTERLACED__ > 1) { y(2,1) = _mm_set1_epi32(0); }
#if __XSHA1_INTERLACED__ > 1
				for (x2 = 0; x2 < 32; x2++, y(2,1) = y(2,0) = _mm_add_epi32(y(2,0), _mm_set1_epi32(1)))
#elif __XSHA1_INTERLACED__ > 0
				for (x2 = 0; x2 < 32; x2++,          y(2,0) = _mm_add_epi32(y(2,0), _mm_set1_epi32(1)))
#endif
				{
					y0 = _mm_set1_epi32(0);
					for (x0 = 0; /*x0 < 5*/; x0++, y0 = _mm_add_epi32(y0, _mm_set1_epi32(1)))
					{
						if (__XSHA1_INTERLACED__ > 0) { y( 0,0) = _mm_xor_si128(y(2,0), y0); }
						if (__XSHA1_INTERLACED__ > 1) { y( 0,1) = y( 0,0); }
						if (__XSHA1_INTERLACED__ > 0) { y(16,0) = POW_2_OF_X(y0); }
						if (__XSHA1_INTERLACED__ > 1) { y(16,1) = y(16,0); }
						if (__XSHA1_INTERLACED__ > 0) { y(17,0) = POW_2_OF_X(y1); }
						if (__XSHA1_INTERLACED__ > 1) { y(17,1) = y(17,0); }
						if (__XSHA1_INTERLACED__ > 0) { y(18,0) = POW_2_OF_X(                            _mm_xor_si128(y( 2,0), y( 4,0))); }
						if (__XSHA1_INTERLACED__ > 1) { y(18,1) = POW_2_OF_X(                            _mm_xor_si128(y( 2,1), y( 4,1))); }
						if (__XSHA1_INTERLACED__ > 0) { y(19,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(y( 3,0),                     y(16,0)), _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 1) { y(19,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(y( 3,1),                     y(16,1)), _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 0) { y(20,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(y( 4,0),                     y(17,0)), _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 1) { y(20,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(y( 4,1),                     y(17,1)), _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 0) { y(21,0) = POW_2_OF_X(_mm_and_si128(                                                         y(18,0) , _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 1) { y(21,1) = POW_2_OF_X(_mm_and_si128(                                                         y(18,1) , _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 0) { y(22,0) = POW_2_OF_X(_mm_and_si128(                                                         y(19,0) , _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 1) { y(22,1) = POW_2_OF_X(_mm_and_si128(                                                         y(19,1) , _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 0) { y(23,0) = POW_2_OF_X(_mm_and_si128(                                                         y(20,0) , _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 1) { y(23,1) = POW_2_OF_X(_mm_and_si128(                                                         y(20,1) , _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 0) { y(24,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(16,0) , y(21,0)), _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 1) { y(24,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(16,1) , y(21,1)), _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 0) { y(25,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(17,0) , y(22,0)), _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 1) { y(25,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(17,1) , y(22,1)), _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 0) { y(26,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(18,0) , y(23,0)), _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 1) { y(26,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(18,1) , y(23,1)), _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 0) { y(27,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(19,0) , y(24,0)), _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 1) { y(27,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(19,1) , y(24,1)), _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 0) { y(28,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(20,0) , y(25,0)), _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 1) { y(28,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(20,1) , y(25,1)), _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 0) { y(29,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(21,0) , y(26,0)), _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 1) { y(29,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(21,1) , y(26,1)), _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 0) { y(30,0) = POW_2_OF_X(_mm_and_si128(_mm_xor_si128(_mm_xor_si128(         y(16,0) , y(22,0)), y(27,0)), _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 1) { y(30,1) = POW_2_OF_X(_mm_and_si128(_mm_xor_si128(_mm_xor_si128(         y(16,1) , y(22,1)), y(27,1)), _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 0) { y(31,0) = POW_2_OF_X(_mm_and_si128(_mm_xor_si128(_mm_xor_si128(         y(17,0) , y(23,0)), y(28,0)), _mm_set1_epi32(31))); }
						if (__XSHA1_INTERLACED__ > 1) { y(31,1) = POW_2_OF_X(_mm_and_si128(_mm_xor_si128(_mm_xor_si128(         y(17,1) , y(23,1)), y(28,1)), _mm_set1_epi32(31))); }
						EXPANSION(y,16); EXPANSION(y,17); EXPANSION(y,18); EXPANSION(y,19);
						EXPANSION(y,20); EXPANSION(y,21); EXPANSION(y,22); EXPANSION(y,23);
						EXPANSION(y,24); EXPANSION(y,25); EXPANSION(y,26); EXPANSION(y,27);
						EXPANSION(y,28); EXPANSION(y,29); EXPANSION(y,30); EXPANSION(y,31);
						EXPANSION(y,32); EXPANSION(y,33); EXPANSION(y,34); EXPANSION(y,35);
						EXPANSION(y,36); EXPANSION(y,37); EXPANSION(y,38); EXPANSION(y,39);
						EXPANSION(y,40); EXPANSION(y,41); EXPANSION(y,42); EXPANSION(y,43);
						EXPANSION(y,44); EXPANSION(y,45); EXPANSION(y,46); EXPANSION(y,47);
						EXPANSION(y,48); EXPANSION(y,49); EXPANSION(y,50); EXPANSION(y,51);
						EXPANSION(y,52); EXPANSION(y,53); EXPANSION(y,54); EXPANSION(y,55);
						EXPANSION(y,56); EXPANSION(y,57); EXPANSION(y,58); EXPANSION(y,59);
						EXPANSION(y,60); EXPANSION(y,61); EXPANSION(y,62); EXPANSION(y,63);

						if (__XSHA1_INTERLACED__ > 0) { a[0] = hash[0]; }
						if (__XSHA1_INTERLACED__ > 1) { a[1] = hash[0]; }
						if (__XSHA1_INTERLACED__ > 0) { b[0] = hash[1]; }
						if (__XSHA1_INTERLACED__ > 1) { b[1] = hash[1]; }
						if (__XSHA1_INTERLACED__ > 0) { c[0] = hash[2]; }
						if (__XSHA1_INTERLACED__ > 1) { c[1] = hash[2]; }
						if (__XSHA1_INTERLACED__ > 0) { d[0] = hash[3]; }
						if (__XSHA1_INTERLACED__ > 1) { d[1] = hash[3]; }
						if (__XSHA1_INTERLACED__ > 0) { e[0] = hash[4]; }
						if (__XSHA1_INTERLACED__ > 1) { e[1] = hash[4]; }

						SHA1_II_REV(b,c,d,e,a,    y,79); SHA1_II_REV(c,d,e,a,b,    y,78); SHA1_II_REV(d,e,a,b,c,    y,77); SHA1_II_REV(e,a,b,c,d,    y,76); SHA1_II_REV(a,b,c,d,e,    y,75);
						SHA1_II_REV(b,c,d,e,a,    y,74); SHA1_II_REV(c,d,e,a,b,    y,73); SHA1_II_REV(d,e,a,b,c,    y,72); SHA1_II_REV(e,a,b,c,d,    y,71); SHA1_II_REV(a,b,c,d,e,    y,70);
						SHA1_II_REV(b,c,d,e,a,    y,69); SHA1_II_REV(c,d,e,a,b,    y,68); SHA1_II_REV(d,e,a,b,c,    y,67); SHA1_II_REV(e,a,b,c,d,    y,66); SHA1_II_REV(a,b,c,d,e,    y,65);
						SHA1_II_REV(b,c,d,e,a,    y,64); SHA1_II_REV(c,d,e,a,b,    y,63); SHA1_II_REV(d,e,a,b,c,    y,62); SHA1_II_REV(e,a,b,c,d,    y,61); SHA1_II_REV(a,b,c,d,e,    y,60);

						SHA1_HH_REV(b,c,d,e,a,    y,59); SHA1_HH_REV(c,d,e,a,b,    y,58); SHA1_HH_REV(d,e,a,b,c,    y,57); SHA1_HH_REV(e,a,b,c,d,    y,56); SHA1_HH_REV(a,b,c,d,e,    y,55);
						SHA1_HH_REV(b,c,d,e,a,    y,54); SHA1_HH_REV(c,d,e,a,b,    y,53); SHA1_HH_REV(d,e,a,b,c,    y,52); SHA1_HH_REV(e,a,b,c,d,    y,51); SHA1_HH_REV(a,b,c,d,e,    y,50);
						SHA1_HH_REV(b,c,d,e,a,    y,49); SHA1_HH_REV(c,d,e,a,b,    y,48); SHA1_HH_REV(d,e,a,b,c,    y,47); SHA1_HH_REV(e,a,b,c,d,    y,46); SHA1_HH_REV(a,b,c,d,e,    y,45);
						SHA1_HH_REV(b,c,d,e,a,    y,44); SHA1_HH_REV(c,d,e,a,b,    y,43); SHA1_HH_REV(d,e,a,b,c,    y,42); SHA1_HH_REV(e,a,b,c,d,    y,41); SHA1_HH_REV(a,b,c,d,e,    y,40);

						SHA1_GG_REV(b,c,d,e,a,    y,39); SHA1_GG_REV(c,d,e,a,b,    y,38); SHA1_GG_REV(d,e,a,b,c,    y,37); SHA1_GG_REV(e,a,b,c,d,    y,36); SHA1_GG_REV(a,b,c,d,e,    y,35);
						SHA1_GG_REV(b,c,d,e,a,    y,34); SHA1_GG_REV(c,d,e,a,b,    y,33); SHA1_GG_REV(d,e,a,b,c,    y,32); SHA1_GG_REV(e,a,b,c,d,    y,31); SHA1_GG_REV(a,b,c,d,e,    y,30);
						SHA1_GG_REV(b,c,d,e,a,    y,29); SHA1_GG_REV(c,d,e,a,b,    y,28); SHA1_GG_REV(d,e,a,b,c,    y,27); SHA1_GG_REV(e,a,b,c,d,    y,26); SHA1_GG_REV(a,b,c,d,e,    y,25);
						SHA1_GG_REV(b,c,d,e,a,    y,24); SHA1_GG_REV(c,d,e,a,b,    y,23); SHA1_GG_REV(d,e,a,b,c,    y,22); SHA1_GG_REV(e,a,b,c,d,    y,21); SHA1_GG_REV(a,b,c,d,e,    y,20);

						SHA1_FF_REV(b,c,d,e,a,    y,19); SHA1_FF_REV(c,d,e,a,b,    y,18); SHA1_FF_REV(d,e,a,b,c,    y,17);

						if (x0 == 5)
						{
							break;
						}

						SHA1_FF_REV(e,a,b,c,d,    y,16);

						SHA1_FF_REV(a,b,c,d,e,ZEROY,15);
						SHA1_FF_REV(b,c,d,e,a,ZEROY,14); SHA1_FF_REV(c,d,e,a,b,ZEROY,13); SHA1_FF_REV(d,e,a,b,c,ZEROY,12); SHA1_FF_REV(e,a,b,c,d,ZEROY,11); SHA1_FF_REV(a,b,c,d,e,ZEROY,10);
						SHA1_FF_REV(b,c,d,e,a,ZEROY, 9); SHA1_FF_REV(c,d,e,a,b,ZEROY, 8); SHA1_FF_REV(d,e,a,b,c,ZEROY, 7); SHA1_FF_REV(e,a,b,c,d,ZEROY, 6); SHA1_FF_REV(a,b,c,d,e,ZEROY, 5);

						if (__XSHA1_INTERLACED__ > 0)
						{
							//x[16] = ROT(e[0], 32 - 30) - (0xc3d2e1f0 + ((0xefcdab89 & 0x98badcfe) | ((~0xefcdab89) & 0x10325476)) + ROT(0x67452301, 5) + 0x5a827999);
							tmp = _mm_sub_epi32(ROT(e[0], 32 - 30), _mm_set1_epi32(0x9FB498B3));
							stmp = _mm_movemask_epi8(_mm_cmpeq_epi32(_mm_and_si128(tmp, _mm_set1_epi32(31)), y(0,0)));
							while (stmp != 0)
							{
								if (finishRev20(&tmp, a + 0, b + 0, c + 0, d + 0, e + 0, sx, x1, x2, x3, x4, stmp))
								{
									invalid = outputPassword(sx, onlyValid, onlyPrintable, h, fout);
									invalidCount += invalid;
									count++;
									if (invalid == 0 && noCollisions != 0)
									{
										return count - invalidCount;
									}
								}
							}
						}
						if (__XSHA1_INTERLACED__ > 1)
						{
							//x[16] = ROT(e[1], 32 - 30) - (0xc3d2e1f0 + ((0xefcdab89 & 0x98badcfe) | ((~0xefcdab89) & 0x10325476)) + ROT(0x67452301, 5) + 0x5a827999);
							tmp = _mm_sub_epi32(ROT(e[1], 32 - 30), _mm_set1_epi32(0x9FB498B3));
							stmp = _mm_movemask_epi8(_mm_cmpeq_epi32(_mm_and_si128(tmp, _mm_set1_epi32(31)), y(0,1)));
							while (stmp != 0)
							{
								if (finishRev20(&tmp, a + 1, b + 1, c + 1, d + 1, e + 1, sx, x1, x2, x3, x4 + 4, stmp))
								{
									invalid = outputPassword(sx, onlyValid, onlyPrintable, h, fout);
									invalidCount += invalid;
									count++;
									if (invalid == 0 && noCollisions != 0)
									{
										return count - invalidCount;
									}
								}
							}
						}
					}

					SHA1_FF_REV(e,a,b,c,d,ZEROY,16);
					if (__XSHA1_INTERLACED__ > 0) { t16a[0] = a[0]; }
					if (__XSHA1_INTERLACED__ > 1) { t16a[1] = a[1]; }
					if (__XSHA1_INTERLACED__ > 0) { t16b[0] = b[0]; }
					if (__XSHA1_INTERLACED__ > 1) { t16b[1] = b[1]; }
					if (__XSHA1_INTERLACED__ > 0) { t16c[0] = c[0]; }
					if (__XSHA1_INTERLACED__ > 1) { t16c[1] = c[1]; }
					if (__XSHA1_INTERLACED__ > 0) { t16d[0] = d[0]; }
					if (__XSHA1_INTERLACED__ > 1) { t16d[1] = d[1]; }
					if (__XSHA1_INTERLACED__ > 0) { t16e[0] = e[0]; }
					if (__XSHA1_INTERLACED__ > 1) { t16e[1] = e[1]; }
					// x[16] = (1 << 5) to (1 << 31)
#if __XSHA1_INTERLACED__ > 1
					for (; x0 < 32; x0++, y(16,1) = y(16,0) = _mm_slli_epi32(y(16,0), 1), y(0,1) = y(0,0) = _mm_xor_si128(y0 = _mm_add_epi32(y0, _mm_set1_epi32(1)), y(2,0)))
#elif __XSHA1_INTERLACED__ > 0
					for (; x0 < 32; x0++,           y(16,0) = _mm_slli_epi32(y(16,0), 1),          y(0,0) = _mm_xor_si128(y0 = _mm_add_epi32(y0, _mm_set1_epi32(1)), y(2,0)))
#endif
					{
						if (__XSHA1_INTERLACED__ > 0) { a[0] = t16a[0]; }
						if (__XSHA1_INTERLACED__ > 1) { a[1] = t16a[1]; }
						if (__XSHA1_INTERLACED__ > 0) { b[0] = t16b[0]; }
						if (__XSHA1_INTERLACED__ > 1) { b[1] = t16b[1]; }
						if (__XSHA1_INTERLACED__ > 0) { c[0] = t16c[0]; }
						if (__XSHA1_INTERLACED__ > 1) { c[1] = t16c[1]; }
						if (__XSHA1_INTERLACED__ > 0) { d[0] = _mm_sub_epi32(t16d[0], y(16,0)); }
						if (__XSHA1_INTERLACED__ > 1) { d[1] = _mm_sub_epi32(t16d[1], y(16,1)); }
						if (__XSHA1_INTERLACED__ > 0) { e[0] = t16e[0]; }
						if (__XSHA1_INTERLACED__ > 1) { e[1] = t16e[1]; }

						SHA1_FF_REV(a,b,c,d,e,ZEROY,15);
						SHA1_FF_REV(b,c,d,e,a,ZEROY,14); SHA1_FF_REV(c,d,e,a,b,ZEROY,13); SHA1_FF_REV(d,e,a,b,c,ZEROY,12); SHA1_FF_REV(e,a,b,c,d,ZEROY,11); SHA1_FF_REV(a,b,c,d,e,ZEROY,10);
						SHA1_FF_REV(b,c,d,e,a,ZEROY, 9); SHA1_FF_REV(c,d,e,a,b,ZEROY, 8); SHA1_FF_REV(d,e,a,b,c,ZEROY, 7); SHA1_FF_REV(e,a,b,c,d,ZEROY, 6); SHA1_FF_REV(a,b,c,d,e,ZEROY, 5);

						if (__XSHA1_INTERLACED__ > 0)
						{
							//x[16] = ROT(e[0], 32 - 30) - (0xc3d2e1f0 + ((0xefcdab89 & 0x98badcfe) | ((~0xefcdab89) & 0x10325476)) + ROT(0x67452301, 5) + 0x5a827999);
							tmp = _mm_sub_epi32(ROT(e[0], 32 - 30), _mm_set1_epi32(0x9FB498B3));
							stmp = _mm_movemask_epi8(_mm_cmpeq_epi32(_mm_and_si128(tmp, _mm_set1_epi32(31)), y(0,0)));
							while (stmp != 0)
							{
								if (finishRev20(&tmp, a + 0, b + 0, c + 0, d + 0, e + 0, sx, x1, x2, x3, x4, stmp))
								{
									invalid = outputPassword(sx, onlyValid, onlyPrintable, h, fout);
									invalidCount += invalid;
									count++;
									if (invalid == 0 && noCollisions != 0)
									{
										return count - invalidCount;
									}
								}
							}
						}
						if (__XSHA1_INTERLACED__ > 1)
						{
							//x[16] = ROT(e[1], 32 - 30) - (0xc3d2e1f0 + ((0xefcdab89 & 0x98badcfe) | ((~0xefcdab89) & 0x10325476)) + ROT(0x67452301, 5) + 0x5a827999);
							tmp = _mm_sub_epi32(ROT(e[1], 32 - 30), _mm_set1_epi32(0x9FB498B3));
							stmp = _mm_movemask_epi8(_mm_cmpeq_epi32(_mm_and_si128(tmp, _mm_set1_epi32(31)), y(0,1)));
							while (stmp != 0)
							{
								if (finishRev20(&tmp, a + 1, b + 1, c + 1, d + 1, e + 1, sx, x1, x2, x3, x4 + 4, stmp))
								{
									invalid = outputPassword(sx, onlyValid, onlyPrintable, h, fout);
									invalidCount += invalid;
									count++;
									if (invalid == 0 && noCollisions != 0)
									{
										return count - invalidCount;
									}
								}
							}
						}
					}
				}
			}
			if (__XSHA1_INTERLACED__ > 0) { y(1,0) = _mm_xor_si128(y(3,0), _mm_set1_epi32(5)); }
			if (__XSHA1_INTERLACED__ > 1) { y(1,1) = _mm_xor_si128(y(3,1), _mm_set1_epi32(5)); }
			if (__XSHA1_INTERLACED__ > 0) { y(2,0) = _mm_set1_epi32(0); }
			if (__XSHA1_INTERLACED__ > 1) { y(2,1) = _mm_set1_epi32(0); }
#if __XSHA1_INTERLACED__ > 1
			for (x2 = 0; x2 < 32; x2++, y(2,1) = y(2,0) = _mm_add_epi32(y(2,0), _mm_set1_epi32(1)))
#elif __XSHA1_INTERLACED__ > 0
			for (x2 = 0; x2 < 32; x2++,          y(2,0) = _mm_add_epi32(y(2,0), _mm_set1_epi32(1)))
#endif
			{
				y0 = _mm_set1_epi32(0);
				for (x0 = 0; /*x0 < 5*/; x0++, y0 = _mm_add_epi32(y0, _mm_set1_epi32(1)))
				{
					if (__XSHA1_INTERLACED__ > 0) { y( 0,0) = _mm_xor_si128(y(2,0), y0); }
					if (__XSHA1_INTERLACED__ > 1) { y( 0,1) = y( 0,0); }
					if (__XSHA1_INTERLACED__ > 0) { y(16,0) = POW_2_OF_X(y0); }
					if (__XSHA1_INTERLACED__ > 1) { y(16,1) = y(16,0); }
					if (__XSHA1_INTERLACED__ > 0) { y(17,0) = _mm_set1_epi32(1 << 5); }
					if (__XSHA1_INTERLACED__ > 1) { y(17,1) = _mm_set1_epi32(1 << 5); }
					if (__XSHA1_INTERLACED__ > 0) { y(18,0) = POW_2_OF_X(                            _mm_xor_si128(y( 2,0), y( 4,0))); }
					if (__XSHA1_INTERLACED__ > 1) { y(18,1) = POW_2_OF_X(                            _mm_xor_si128(y( 2,1), y( 4,1))); }
					if (__XSHA1_INTERLACED__ > 0) { y(19,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(y( 3,0),                     y(16,0)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(19,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(y( 3,1),                     y(16,1)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 0) { y(20,0) = POW_2_OF_X(y( 4,0)); }
					if (__XSHA1_INTERLACED__ > 1) { y(20,1) = POW_2_OF_X(y( 4,1)); }
					if (__XSHA1_INTERLACED__ > 0) { y(21,0) = POW_2_OF_X(_mm_and_si128(                                                         y(18,0) , _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(21,1) = POW_2_OF_X(_mm_and_si128(                                                         y(18,1) , _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 0) { y(22,0) = POW_2_OF_X(_mm_and_si128(                                                         y(19,0) , _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(22,1) = POW_2_OF_X(_mm_and_si128(                                                         y(19,1) , _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 0) { y(23,0) = POW_2_OF_X(_mm_and_si128(                                                         y(20,0) , _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(23,1) = POW_2_OF_X(_mm_and_si128(                                                         y(20,1) , _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 0) { y(24,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(16,0) , y(21,0)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(24,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(16,1) , y(21,1)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 0) { y(25,0) = POW_2_OF_X(_mm_and_si128(                                                         y(22,0) , _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(25,1) = POW_2_OF_X(_mm_and_si128(                                                         y(22,1) , _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 0) { y(26,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(18,0) , y(23,0)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(26,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(18,1) , y(23,1)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 0) { y(27,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(19,0) , y(24,0)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(27,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(19,1) , y(24,1)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 0) { y(28,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(20,0) , y(25,0)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(28,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(20,1) , y(25,1)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 0) { y(29,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(21,0) , y(26,0)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(29,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(21,1) , y(26,1)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 0) { y(30,0) = POW_2_OF_X(_mm_and_si128(_mm_xor_si128(_mm_xor_si128(         y(16,0) , y(22,0)), y(27,0)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(30,1) = POW_2_OF_X(_mm_and_si128(_mm_xor_si128(_mm_xor_si128(         y(16,1) , y(22,1)), y(27,1)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 0) { y(31,0) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(23,0) , y(28,0)), _mm_set1_epi32(31))); }
					if (__XSHA1_INTERLACED__ > 1) { y(31,1) = POW_2_OF_X(_mm_and_si128(              _mm_xor_si128(                   y(23,1) , y(28,1)), _mm_set1_epi32(31))); }
					EXPANSION(y,16); EXPANSION(y,17); EXPANSION(y,18); EXPANSION(y,19);
					EXPANSION(y,20); EXPANSION(y,21); EXPANSION(y,22); EXPANSION(y,23);
					EXPANSION(y,24); EXPANSION(y,25); EXPANSION(y,26); EXPANSION(y,27);
					EXPANSION(y,28); EXPANSION(y,29); EXPANSION(y,30); EXPANSION(y,31);
					EXPANSION(y,32); EXPANSION(y,33); EXPANSION(y,34); EXPANSION(y,35);
					EXPANSION(y,36); EXPANSION(y,37); EXPANSION(y,38); EXPANSION(y,39);
					EXPANSION(y,40); EXPANSION(y,41); EXPANSION(y,42); EXPANSION(y,43);
					EXPANSION(y,44); EXPANSION(y,45); EXPANSION(y,46); EXPANSION(y,47);
					EXPANSION(y,48); EXPANSION(y,49); EXPANSION(y,50); EXPANSION(y,51);
					EXPANSION(y,52); EXPANSION(y,53); EXPANSION(y,54); EXPANSION(y,55);
					EXPANSION(y,56); EXPANSION(y,57); EXPANSION(y,58); EXPANSION(y,59);
					EXPANSION(y,60); EXPANSION(y,61); EXPANSION(y,62); EXPANSION(y,63);

					if (__XSHA1_INTERLACED__ > 0) { a[0] = hash[0]; }
					if (__XSHA1_INTERLACED__ > 1) { a[1] = hash[0]; }
					if (__XSHA1_INTERLACED__ > 0) { b[0] = hash[1]; }
					if (__XSHA1_INTERLACED__ > 1) { b[1] = hash[1]; }
					if (__XSHA1_INTERLACED__ > 0) { c[0] = hash[2]; }
					if (__XSHA1_INTERLACED__ > 1) { c[1] = hash[2]; }
					if (__XSHA1_INTERLACED__ > 0) { d[0] = hash[3]; }
					if (__XSHA1_INTERLACED__ > 1) { d[1] = hash[3]; }
					if (__XSHA1_INTERLACED__ > 0) { e[0] = hash[4]; }
					if (__XSHA1_INTERLACED__ > 1) { e[1] = hash[4]; }

					SHA1_II_REV(b,c,d,e,a,    y,79); SHA1_II_REV(c,d,e,a,b,    y,78); SHA1_II_REV(d,e,a,b,c,    y,77); SHA1_II_REV(e,a,b,c,d,    y,76); SHA1_II_REV(a,b,c,d,e,    y,75);
					SHA1_II_REV(b,c,d,e,a,    y,74); SHA1_II_REV(c,d,e,a,b,    y,73); SHA1_II_REV(d,e,a,b,c,    y,72); SHA1_II_REV(e,a,b,c,d,    y,71); SHA1_II_REV(a,b,c,d,e,    y,70);
					SHA1_II_REV(b,c,d,e,a,    y,69); SHA1_II_REV(c,d,e,a,b,    y,68); SHA1_II_REV(d,e,a,b,c,    y,67); SHA1_II_REV(e,a,b,c,d,    y,66); SHA1_II_REV(a,b,c,d,e,    y,65);
					SHA1_II_REV(b,c,d,e,a,    y,64); SHA1_II_REV(c,d,e,a,b,    y,63); SHA1_II_REV(d,e,a,b,c,    y,62); SHA1_II_REV(e,a,b,c,d,    y,61); SHA1_II_REV(a,b,c,d,e,    y,60);

					SHA1_HH_REV(b,c,d,e,a,    y,59); SHA1_HH_REV(c,d,e,a,b,    y,58); SHA1_HH_REV(d,e,a,b,c,    y,57); SHA1_HH_REV(e,a,b,c,d,    y,56); SHA1_HH_REV(a,b,c,d,e,    y,55);
					SHA1_HH_REV(b,c,d,e,a,    y,54); SHA1_HH_REV(c,d,e,a,b,    y,53); SHA1_HH_REV(d,e,a,b,c,    y,52); SHA1_HH_REV(e,a,b,c,d,    y,51); SHA1_HH_REV(a,b,c,d,e,    y,50);
					SHA1_HH_REV(b,c,d,e,a,    y,49); SHA1_HH_REV(c,d,e,a,b,    y,48); SHA1_HH_REV(d,e,a,b,c,    y,47); SHA1_HH_REV(e,a,b,c,d,    y,46); SHA1_HH_REV(a,b,c,d,e,    y,45);
					SHA1_HH_REV(b,c,d,e,a,    y,44); SHA1_HH_REV(c,d,e,a,b,    y,43); SHA1_HH_REV(d,e,a,b,c,    y,42); SHA1_HH_REV(e,a,b,c,d,    y,41); SHA1_HH_REV(a,b,c,d,e,    y,40);

					SHA1_GG_REV(b,c,d,e,a,    y,39); SHA1_GG_REV(c,d,e,a,b,    y,38); SHA1_GG_REV(d,e,a,b,c,    y,37); SHA1_GG_REV(e,a,b,c,d,    y,36); SHA1_GG_REV(a,b,c,d,e,    y,35);
					SHA1_GG_REV(b,c,d,e,a,    y,34); SHA1_GG_REV(c,d,e,a,b,    y,33); SHA1_GG_REV(d,e,a,b,c,    y,32); SHA1_GG_REV(e,a,b,c,d,    y,31); SHA1_GG_REV(a,b,c,d,e,    y,30);
					SHA1_GG_REV(b,c,d,e,a,    y,29); SHA1_GG_REV(c,d,e,a,b,    y,28); SHA1_GG_REV(d,e,a,b,c,    y,27); SHA1_GG_REV(e,a,b,c,d,    y,26); SHA1_GG_REV(a,b,c,d,e,    y,25);
					SHA1_GG_REV(b,c,d,e,a,    y,24); SHA1_GG_REV(c,d,e,a,b,    y,23); SHA1_GG_REV(d,e,a,b,c,    y,22); SHA1_GG_REV(e,a,b,c,d,    y,21); SHA1_GG_REV(a,b,c,d,e,    y,20);

					SHA1_FF_REV(b,c,d,e,a,    y,19); SHA1_FF_REV(c,d,e,a,b,    y,18);

					SHA1_FF_REV(d,e,a,b,c,ZEROY,17);
					if (__XSHA1_INTERLACED__ > 0) { t17a[0] = a[0]; }
					if (__XSHA1_INTERLACED__ > 1) { t17a[1] = a[1]; }
					if (__XSHA1_INTERLACED__ > 0) { t17b[0] = b[0]; }
					if (__XSHA1_INTERLACED__ > 1) { t17b[1] = b[1]; }
					if (__XSHA1_INTERLACED__ > 0) { t17c[0] = c[0]; }
					if (__XSHA1_INTERLACED__ > 1) { t17c[1] = c[1]; }
					if (__XSHA1_INTERLACED__ > 0) { t17d[0] = d[0]; }
					if (__XSHA1_INTERLACED__ > 1) { t17d[1] = d[1]; }
					if (__XSHA1_INTERLACED__ > 0) { t17e[0] = e[0]; }
					if (__XSHA1_INTERLACED__ > 1) { t17e[1] = e[1]; }

					if (x0 == 5)
					{
						break;
					}

					// x[16] = (1 << 5) to (1 << 31)
#if __XSHA1_INTERLACED__ > 1
					for (; x1 < 32; x1++, y(17,1) = y(17,0) = _mm_slli_epi32(y(17,0), 1), y(1,1) = y(1,0) = _mm_xor_si128(y1 = _mm_add_epi32(y1, _mm_set1_epi32(1)), y(3,0)))
#elif __XSHA1_INTERLACED__ > 0
					for (; x1 < 32; x1++,           y(17,0) = _mm_slli_epi32(y(17,0), 1),          y(1,0) = _mm_xor_si128(y1 = _mm_add_epi32(y1, _mm_set1_epi32(1)), y(3,0)))
#endif
					{
						if (__XSHA1_INTERLACED__ > 0) { a[0] = t17a[0]; }
						if (__XSHA1_INTERLACED__ > 1) { a[1] = t17a[1]; }
						if (__XSHA1_INTERLACED__ > 0) { b[0] = t17b[0]; }
						if (__XSHA1_INTERLACED__ > 1) { b[1] = t17b[1]; }
						if (__XSHA1_INTERLACED__ > 0) { c[0] = _mm_sub_epi32(t17c[0], y(17,0)); }
						if (__XSHA1_INTERLACED__ > 1) { c[1] = _mm_sub_epi32(t17c[1], y(17,1)); }
						if (__XSHA1_INTERLACED__ > 0) { d[0] = t17d[0]; }
						if (__XSHA1_INTERLACED__ > 1) { d[1] = t17d[1]; }
						if (__XSHA1_INTERLACED__ > 0) { e[0] = t17e[0]; }
						if (__XSHA1_INTERLACED__ > 1) { e[1] = t17e[1]; }

						SHA1_FF_REV(e,a,b,c,d,    y,16);

						SHA1_FF_REV(a,b,c,d,e,ZEROY,15);
						SHA1_FF_REV(b,c,d,e,a,ZEROY,14); SHA1_FF_REV(c,d,e,a,b,ZEROY,13); SHA1_FF_REV(d,e,a,b,c,ZEROY,12); SHA1_FF_REV(e,a,b,c,d,ZEROY,11); SHA1_FF_REV(a,b,c,d,e,ZEROY,10);
						SHA1_FF_REV(b,c,d,e,a,ZEROY, 9); SHA1_FF_REV(c,d,e,a,b,ZEROY, 8); SHA1_FF_REV(d,e,a,b,c,ZEROY, 7); SHA1_FF_REV(e,a,b,c,d,ZEROY, 6); SHA1_FF_REV(a,b,c,d,e,ZEROY, 5);

						if (__XSHA1_INTERLACED__ > 0)
						{
							//x[16] = ROT(e[0], 32 - 30) - (0xc3d2e1f0 + ((0xefcdab89 & 0x98badcfe) | ((~0xefcdab89) & 0x10325476)) + ROT(0x67452301, 5) + 0x5a827999);
							tmp = _mm_sub_epi32(ROT(e[0], 32 - 30), _mm_set1_epi32(0x9FB498B3));
							stmp = _mm_movemask_epi8(_mm_cmpeq_epi32(_mm_and_si128(tmp, _mm_set1_epi32(31)), y(0,0)));
							while (stmp != 0)
							{
								if (finishRev20(&tmp, a + 0, b + 0, c + 0, d + 0, e + 0, sx, x1, x2, x3, x4, stmp))
								{
									invalid = outputPassword(sx, onlyValid, onlyPrintable, h, fout);
									invalidCount += invalid;
									count++;
									if (invalid == 0 && noCollisions != 0)
									{
										return count - invalidCount;
									}
								}
							}
						}
						if (__XSHA1_INTERLACED__ > 1)
						{
							//x[16] = ROT(e[1], 32 - 30) - (0xc3d2e1f0 + ((0xefcdab89 & 0x98badcfe) | ((~0xefcdab89) & 0x10325476)) + ROT(0x67452301, 5) + 0x5a827999);
							tmp = _mm_sub_epi32(ROT(e[1], 32 - 30), _mm_set1_epi32(0x9FB498B3));
							stmp = _mm_movemask_epi8(_mm_cmpeq_epi32(_mm_and_si128(tmp, _mm_set1_epi32(31)), y(0,1)));
							while (stmp != 0)
							{
								if (finishRev20(&tmp, a + 1, b + 1, c + 1, d + 1, e + 1, sx, x1, x2, x3, x4 + 4, stmp))
								{
									invalid = outputPassword(sx, onlyValid, onlyPrintable, h, fout);
									invalidCount += invalid;
									count++;
									if (invalid == 0 && noCollisions != 0)
									{
										return count - invalidCount;
									}
								}
							}
						}
					}
					x1 = 5;
					y1 = _mm_set1_epi32(5);
					if (__XSHA1_INTERLACED__ > 0) { y(1,0) = _mm_xor_si128(y(3,0), _mm_set1_epi32(5)); }
					if (__XSHA1_INTERLACED__ > 1) { y(1,1) = _mm_xor_si128(y(3,1), _mm_set1_epi32(5)); }
				}

				// x[17] = (1 << 5) to (1 << 31)
#if __XSHA1_INTERLACED__ > 1
				for (; x1 < 32; x1++, y(17,1) = y(17,0) = _mm_slli_epi32(y(17,0), 1), y(1,1) = y(1,0) = _mm_xor_si128(y1 = _mm_add_epi32(y1, _mm_set1_epi32(1)), y(3,0)))
#elif __XSHA1_INTERLACED__ > 0
				for (; x1 < 32; x1++,           y(17,0) = _mm_slli_epi32(y(17,0), 1),          y(1,0) = _mm_xor_si128(y1 = _mm_add_epi32(y1, _mm_set1_epi32(1)), y(3,0)))
#endif
				{
					if (__XSHA1_INTERLACED__ > 0) { a[0] = t17a[0]; }
					if (__XSHA1_INTERLACED__ > 1) { a[1] = t17a[1]; }
					if (__XSHA1_INTERLACED__ > 0) { b[0] = t17b[0]; }
					if (__XSHA1_INTERLACED__ > 1) { b[1] = t17b[1]; }
					if (__XSHA1_INTERLACED__ > 0) { c[0] = _mm_sub_epi32(t17c[0], y(17,0)); }
					if (__XSHA1_INTERLACED__ > 1) { c[1] = _mm_sub_epi32(t17c[1], y(17,1)); }
					if (__XSHA1_INTERLACED__ > 0) { d[0] = t17d[0]; }
					if (__XSHA1_INTERLACED__ > 1) { d[1] = t17d[1]; }
					if (__XSHA1_INTERLACED__ > 0) { e[0] = t17e[0]; }
					if (__XSHA1_INTERLACED__ > 1) { e[1] = t17e[1]; }

					SHA1_FF_REV(e,a,b,c,d,ZEROY,16);
					if (__XSHA1_INTERLACED__ > 0) { t16a[0] = a[0]; }
					if (__XSHA1_INTERLACED__ > 1) { t16a[1] = a[1]; }
					if (__XSHA1_INTERLACED__ > 0) { t16b[0] = b[0]; }
					if (__XSHA1_INTERLACED__ > 1) { t16b[1] = b[1]; }
					if (__XSHA1_INTERLACED__ > 0) { t16c[0] = c[0]; }
					if (__XSHA1_INTERLACED__ > 1) { t16c[1] = c[1]; }
					if (__XSHA1_INTERLACED__ > 0) { t16d[0] = d[0]; }
					if (__XSHA1_INTERLACED__ > 1) { t16d[1] = d[1]; }
					if (__XSHA1_INTERLACED__ > 0) { t16e[0] = e[0]; }
					if (__XSHA1_INTERLACED__ > 1) { t16e[1] = e[1]; }
					// x[16] = (1 << 5) to (1 << 31)
#if __XSHA1_INTERLACED__ > 1
					for (; x0 < 32; x0++, y(16,1) = y(16,0) = _mm_slli_epi32(y(16,0), 1), y(0,1) = y(0,0) = _mm_xor_si128(y0 = _mm_add_epi32(y0, _mm_set1_epi32(1)), y(2,0)))
#elif __XSHA1_INTERLACED__ > 0
					for (; x0 < 32; x0++,           y(16,0) = _mm_slli_epi32(y(16,0), 1),          y(0,0) = _mm_xor_si128(y0 = _mm_add_epi32(y0, _mm_set1_epi32(1)), y(2,0)))
#endif
					{
						if (__XSHA1_INTERLACED__ > 0) { a[0] = t16a[0]; }
						if (__XSHA1_INTERLACED__ > 1) { a[1] = t16a[1]; }
						if (__XSHA1_INTERLACED__ > 0) { b[0] = t16b[0]; }
						if (__XSHA1_INTERLACED__ > 1) { b[1] = t16b[1]; }
						if (__XSHA1_INTERLACED__ > 0) { c[0] = t16c[0]; }
						if (__XSHA1_INTERLACED__ > 1) { c[1] = t16c[1]; }
						if (__XSHA1_INTERLACED__ > 0) { d[0] = _mm_sub_epi32(t16d[0], y(16,0)); }
						if (__XSHA1_INTERLACED__ > 1) { d[1] = _mm_sub_epi32(t16d[1], y(16,1)); }
						if (__XSHA1_INTERLACED__ > 0) { e[0] = t16e[0]; }
						if (__XSHA1_INTERLACED__ > 1) { e[1] = t16e[1]; }

						SHA1_FF_REV(a,b,c,d,e,ZEROY,15);
						SHA1_FF_REV(b,c,d,e,a,ZEROY,14); SHA1_FF_REV(c,d,e,a,b,ZEROY,13); SHA1_FF_REV(d,e,a,b,c,ZEROY,12); SHA1_FF_REV(e,a,b,c,d,ZEROY,11); SHA1_FF_REV(a,b,c,d,e,ZEROY,10);
						SHA1_FF_REV(b,c,d,e,a,ZEROY, 9); SHA1_FF_REV(c,d,e,a,b,ZEROY, 8); SHA1_FF_REV(d,e,a,b,c,ZEROY, 7); SHA1_FF_REV(e,a,b,c,d,ZEROY, 6); SHA1_FF_REV(a,b,c,d,e,ZEROY, 5);

						if (__XSHA1_INTERLACED__ > 0)
						{
							//x[16] = ROT(e[0], 32 - 30) - (0xc3d2e1f0 + ((0xefcdab89 & 0x98badcfe) | ((~0xefcdab89) & 0x10325476)) + ROT(0x67452301, 5) + 0x5a827999);
							tmp = _mm_sub_epi32(ROT(e[0], 32 - 30), _mm_set1_epi32(0x9FB498B3));
							stmp = _mm_movemask_epi8(_mm_cmpeq_epi32(_mm_and_si128(tmp, _mm_set1_epi32(31)), y(0,0)));
							while (stmp != 0)
							{
								if (finishRev20(&tmp, a + 0, b + 0, c + 0, d + 0, e + 0, sx, x1, x2, x3, x4, stmp))
								{
									invalid = outputPassword(sx, onlyValid, onlyPrintable, h, fout);
									invalidCount += invalid;
									count++;
									if (invalid == 0 && noCollisions != 0)
									{
										return count - invalidCount;
									}
								}
							}
						}
						if (__XSHA1_INTERLACED__ > 1)
						{
							//x[16] = ROT(e[1], 32 - 30) - (0xc3d2e1f0 + ((0xefcdab89 & 0x98badcfe) | ((~0xefcdab89) & 0x10325476)) + ROT(0x67452301, 5) + 0x5a827999);
							tmp = _mm_sub_epi32(ROT(e[1], 32 - 30), _mm_set1_epi32(0x9FB498B3));
							stmp = _mm_movemask_epi8(_mm_cmpeq_epi32(_mm_and_si128(tmp, _mm_set1_epi32(31)), y(0,1)));
							while (stmp != 0)
							{
								if (finishRev20(&tmp, a + 1, b + 1, c + 1, d + 1, e + 1, sx, x1, x2, x3, x4 + 4, stmp))
								{
									invalid = outputPassword(sx, onlyValid, onlyPrintable, h, fout);
									invalidCount += invalid;
									count++;
									if (invalid == 0 && noCollisions != 0)
									{
										return count - invalidCount;
									}
								}
							}
						}
					}
					x0 = 5;
					y0 = _mm_set1_epi32(5);
					if (__XSHA1_INTERLACED__ > 0) { y( 0,0) = _mm_xor_si128(y(2,0), _mm_set1_epi32(5)); }
					if (__XSHA1_INTERLACED__ > 1) { y( 0,1) = _mm_xor_si128(y(2,1), _mm_set1_epi32(5)); }
					if (__XSHA1_INTERLACED__ > 0) { y(16,0) = _mm_set1_epi32(1 << 5); }
					if (__XSHA1_INTERLACED__ > 1) { y(16,1) = _mm_set1_epi32(1 << 5); }
				}
				x1 = 5;
				y1 = _mm_set1_epi32(5);
				if (__XSHA1_INTERLACED__ > 0) { y(1,0) = _mm_xor_si128(y(3,0), _mm_set1_epi32(5)); }
				if (__XSHA1_INTERLACED__ > 1) { y(1,1) = _mm_xor_si128(y(3,1), _mm_set1_epi32(5)); }
			}
		}
	}
	return count - invalidCount;
}
#undef ROT
#endif

#endif

#undef LOAD_CONST_SHA1_A
#undef LOAD_CONST_SHA1_B
#undef LOAD_CONST_SHA1_C
#undef LOAD_CONST_SHA1_D
#undef LOAD_CONST_SHA1_E
#undef LOAD_CONST_SHA1_FF
#undef LOAD_CONST_SHA1_GG
#undef LOAD_CONST_SHA1_HH
#undef LOAD_CONST_SHA1_II

#undef ZEROY
#undef y

#undef SHA1_FF_REV_
#undef SHA1_GG_REV_
#undef SHA1_HH_REV_
#undef SHA1_II_REV_

#undef SHA1_FF_REV
#undef SHA1_GG_REV
#undef SHA1_HH_REV
#undef SHA1_II_REV

#undef POW_2_OF_X
#undef EXPANSION_
#undef EXPANSION

#endif
