/*
	Copyright (C) 2012 Steve Thomas <steve AT tobtu DOT com>

	This file is part of XSHA1 Reverser.

	XSHA1 Reverser is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	XSHA1 Reverser is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with XSHA1 Reverser.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "xsha1.h"
#include "xsha1help.h"

#ifdef ARC_x86

uint32_t finishRev16Norm(uint32_t t, uint32_t a, uint32_t b, uint32_t c, uint32_t *x);
uint32_t finishRev20Norm(uint32_t t, uint32_t a, uint32_t b, uint32_t c, uint32_t d, uint32_t *x);

uint32_t XSHA1_rev16::calcSSE2(const uint32_t *hash, uint32_t &invalidCount, bool onlyValid, bool onlyPrintable, bool noCollisions, FILE *fout)
{
	return xsha1Reverse_16(hash, invalidCount, onlyValid, onlyPrintable, noCollisions, fout);
}

uint32_t XSHA1_rev20::calcSSE2(const uint32_t *hash, uint32_t &invalidCount, bool onlyValid, bool onlyPrintable, bool noCollisions, FILE *fout)
{
	return xsha1Reverse_20(hash, invalidCount, onlyValid, onlyPrintable, noCollisions, fout);
}

#endif

#define ROT(n,s) ((n << (s)) | (n >> (32 - (s))))

#define SHA1_FF_REV(a,b,c,d,e,x) b = ROT(b, 2); e -= (((c ^ d) & b) ^ d)       + x + 0x5a827999 + ROT(a, 5);
#define SHA1_GG_REV(a,b,c,d,e,x) b = ROT(b, 2); e -= (b ^ c ^ d)               + x + 0x6ed9eba1 + ROT(a, 5);
#define SHA1_HH_REV(a,b,c,d,e,x) b = ROT(b, 2); e -= (((b | c) & d) | (b & c)) + x + 0x8f1bbcdc + ROT(a, 5);
#define SHA1_II_REV(a,b,c,d,e,x) b = ROT(b, 2); e -= (b ^ c ^ d)               + x + 0xca62c1d6 + ROT(a, 5);
#define EXPANSION(x,i)           x[i + 16] = 1 << ((x[i] ^ x[i + 2] ^ x[i + 8] ^ x[i + 13]) & 31);

uint32_t XSHA1_rev16::calcNorm(const uint32_t *hash, uint32_t &invalidCount, bool onlyValid, bool onlyPrintable, bool noCollisions, FILE *fout)
{
	uint32_t a, b, c, d, e, x0, x1, x[80], count = 0, invalid;
	uint32_t hash0, hash1, hash2, hash3, hash4;
	uint32_t t17a, t17b, t17c, t17d, t17e;
	uint32_t t16a, t16b, t16c, t16d, t16e;

	invalidCount = 0;
	hash0 = hash[0] - 0x67452301;
	hash1 = hash[1] - 0xefcdab89;
	hash2 = hash[2] - 0x98badcfe;
	hash3 = hash[3] - 0x10325476;
	hash4 = hash[4] - 0xc3d2e1f0;

	for (x[3] = 0; x[3] < 32; x[3]++)
	{
		for (x1 = 0; x1 < 5; x1++)
		{
			x[1] = x1 ^ x[3];
			for (x[2] = 0; x[2] < 32; x[2]++)
			{
				for (x0 = 0; /*x0 <= 5*/; x0++)
				{
					x[0] = x0 ^ x[2];
					x[16] = 1 << x0;
					x[17] = 1 << x1;
					x[18] = 1 << x[2];
					x[19] = 1 << ((x[ 3] ^                 x[16]) & 31);
					x[20] = 1 << ((                        x[17]) & 31);
					x[21] = 1 << ((                        x[18]) & 31);
					x[22] = 1 << ((                        x[19]) & 31);
					x[23] = 1 << ((                        x[20]) & 31);
					x[24] = 1 << ((                x[16] ^ x[21]) & 31);
					x[25] = 1 << ((                x[17] ^ x[22]) & 31);
					x[26] = 1 << ((                x[18] ^ x[23]) & 31);
					x[27] = 1 << ((                x[19] ^ x[24]) & 31);
					x[28] = 1 << ((                x[20] ^ x[25]) & 31);
					x[29] = 1 << ((                x[21] ^ x[26]) & 31);
					x[30] = 1 << ((        x[16] ^ x[22] ^ x[27]) & 31);
					x[31] = 1 << ((        x[17] ^ x[23] ^ x[28]) & 31);
					EXPANSION(x,16); EXPANSION(x,17); EXPANSION(x,18); EXPANSION(x,19);
					EXPANSION(x,20); EXPANSION(x,21); EXPANSION(x,22); EXPANSION(x,23);
					EXPANSION(x,24); EXPANSION(x,25); EXPANSION(x,26); EXPANSION(x,27);
					EXPANSION(x,28); EXPANSION(x,29); EXPANSION(x,30); EXPANSION(x,31);
					EXPANSION(x,32); EXPANSION(x,33); EXPANSION(x,34); EXPANSION(x,35);
					EXPANSION(x,36); EXPANSION(x,37); EXPANSION(x,38); EXPANSION(x,39);
					EXPANSION(x,40); EXPANSION(x,41); EXPANSION(x,42); EXPANSION(x,43);
					EXPANSION(x,44); EXPANSION(x,45); EXPANSION(x,46); EXPANSION(x,47);
					EXPANSION(x,48); EXPANSION(x,49); EXPANSION(x,50); EXPANSION(x,51);
					EXPANSION(x,52); EXPANSION(x,53); EXPANSION(x,54); EXPANSION(x,55);
					EXPANSION(x,56); EXPANSION(x,57); EXPANSION(x,58); EXPANSION(x,59);
					EXPANSION(x,60); EXPANSION(x,61); EXPANSION(x,62); EXPANSION(x,63);

					a = hash0;
					b = hash1;
					c = hash2;
					d = hash3;
					e = hash4;

					SHA1_II_REV(b,c,d,e,a,x[79]); SHA1_II_REV(c,d,e,a,b,x[78]); SHA1_II_REV(d,e,a,b,c,x[77]); SHA1_II_REV(e,a,b,c,d,x[76]); SHA1_II_REV(a,b,c,d,e,x[75]);
					SHA1_II_REV(b,c,d,e,a,x[74]); SHA1_II_REV(c,d,e,a,b,x[73]); SHA1_II_REV(d,e,a,b,c,x[72]); SHA1_II_REV(e,a,b,c,d,x[71]); SHA1_II_REV(a,b,c,d,e,x[70]);
					SHA1_II_REV(b,c,d,e,a,x[69]); SHA1_II_REV(c,d,e,a,b,x[68]); SHA1_II_REV(d,e,a,b,c,x[67]); SHA1_II_REV(e,a,b,c,d,x[66]); SHA1_II_REV(a,b,c,d,e,x[65]);
					SHA1_II_REV(b,c,d,e,a,x[64]); SHA1_II_REV(c,d,e,a,b,x[63]); SHA1_II_REV(d,e,a,b,c,x[62]); SHA1_II_REV(e,a,b,c,d,x[61]); SHA1_II_REV(a,b,c,d,e,x[60]);

					SHA1_HH_REV(b,c,d,e,a,x[59]); SHA1_HH_REV(c,d,e,a,b,x[58]); SHA1_HH_REV(d,e,a,b,c,x[57]); SHA1_HH_REV(e,a,b,c,d,x[56]); SHA1_HH_REV(a,b,c,d,e,x[55]);
					SHA1_HH_REV(b,c,d,e,a,x[54]); SHA1_HH_REV(c,d,e,a,b,x[53]); SHA1_HH_REV(d,e,a,b,c,x[52]); SHA1_HH_REV(e,a,b,c,d,x[51]); SHA1_HH_REV(a,b,c,d,e,x[50]);
					SHA1_HH_REV(b,c,d,e,a,x[49]); SHA1_HH_REV(c,d,e,a,b,x[48]); SHA1_HH_REV(d,e,a,b,c,x[47]); SHA1_HH_REV(e,a,b,c,d,x[46]); SHA1_HH_REV(a,b,c,d,e,x[45]);
					SHA1_HH_REV(b,c,d,e,a,x[44]); SHA1_HH_REV(c,d,e,a,b,x[43]); SHA1_HH_REV(d,e,a,b,c,x[42]); SHA1_HH_REV(e,a,b,c,d,x[41]); SHA1_HH_REV(a,b,c,d,e,x[40]);

					SHA1_GG_REV(b,c,d,e,a,x[39]); SHA1_GG_REV(c,d,e,a,b,x[38]); SHA1_GG_REV(d,e,a,b,c,x[37]); SHA1_GG_REV(e,a,b,c,d,x[36]); SHA1_GG_REV(a,b,c,d,e,x[35]);
					SHA1_GG_REV(b,c,d,e,a,x[34]); SHA1_GG_REV(c,d,e,a,b,x[33]); SHA1_GG_REV(d,e,a,b,c,x[32]); SHA1_GG_REV(e,a,b,c,d,x[31]); SHA1_GG_REV(a,b,c,d,e,x[30]);
					SHA1_GG_REV(b,c,d,e,a,x[29]); SHA1_GG_REV(c,d,e,a,b,x[28]); SHA1_GG_REV(d,e,a,b,c,x[27]); SHA1_GG_REV(e,a,b,c,d,x[26]); SHA1_GG_REV(a,b,c,d,e,x[25]);
					SHA1_GG_REV(b,c,d,e,a,x[24]); SHA1_GG_REV(c,d,e,a,b,x[23]); SHA1_GG_REV(d,e,a,b,c,x[22]); SHA1_GG_REV(e,a,b,c,d,x[21]); SHA1_GG_REV(a,b,c,d,e,x[20]);

					SHA1_FF_REV(b,c,d,e,a,x[19]); SHA1_FF_REV(c,d,e,a,b,x[18]); SHA1_FF_REV(d,e,a,b,c,x[17]);

					if (x0 == 5)
					{
						break;
					}

					SHA1_FF_REV(e,a,b,c,d,x[16]);

					SHA1_FF_REV(a,b,c,d,e,    0); // 15
					SHA1_FF_REV(b,c,d,e,a,    0); SHA1_FF_REV(c,d,e,a,b,    0); SHA1_FF_REV(d,e,a,b,c,    0); SHA1_FF_REV(e,a,b,c,d,    0); SHA1_FF_REV(a,b,c,d,e,    0);
					SHA1_FF_REV(b,c,d,e,a,    0); SHA1_FF_REV(c,d,e,a,b,    0); SHA1_FF_REV(d,e,a,b,c,    0); SHA1_FF_REV(e,a,b,c,d,    0); SHA1_FF_REV(a,b,c,d,e,    0);
					SHA1_FF_REV(b,c,d,e,a,    0); // 4

					if (a == 0x59d148c0 && finishRev16Norm(b, c, d, e, x))
					{
						invalid = outputPassword(x + 4, onlyValid, onlyPrintable, hash, fout);
						invalidCount += invalid;
						count++;
						if (invalid == 0 && noCollisions)
						{
							return count - invalidCount;
						}
					}
				}

				SHA1_FF_REV(e,a,b,c,d,0); // 16
				t16a = a;
				t16b = b;
				t16c = c;
				t16d = d;
				t16e = e;
				// x[16] = (1 << 5) to (1 << 31)
				for (; x[16] != 0; x[16] <<= 1, x[0] = ++x0 ^ x[2])
				{
					a = t16a;
					b = t16b;
					c = t16c;
					d = t16d - x[16];
					e = t16e;

					SHA1_FF_REV(a,b,c,d,e,    0); // 15
					SHA1_FF_REV(b,c,d,e,a,    0); SHA1_FF_REV(c,d,e,a,b,    0); SHA1_FF_REV(d,e,a,b,c,    0); SHA1_FF_REV(e,a,b,c,d,    0); SHA1_FF_REV(a,b,c,d,e,    0);
					SHA1_FF_REV(b,c,d,e,a,    0); SHA1_FF_REV(c,d,e,a,b,    0); SHA1_FF_REV(d,e,a,b,c,    0); SHA1_FF_REV(e,a,b,c,d,    0); SHA1_FF_REV(a,b,c,d,e,    0);
					SHA1_FF_REV(b,c,d,e,a,    0); // 4

					if (a == 0x59d148c0 && finishRev16Norm(b, c, d, e, x))
					{
						invalid = outputPassword(x + 4, onlyValid, onlyPrintable, hash, fout);
						invalidCount += invalid;
						count++;
						if (invalid == 0 && noCollisions)
						{
							return count - invalidCount;
						}
					}
				}
			}
		}
		x[1] = 5 ^ x[3];
		for (x[2] = 0; x[2] < 32; x[2]++)
		{
			for (x0 = 0; /*x0 <= 5*/; x0++)
			{
				x[0] = x0 ^ x[2];
				x[16] = 1 << x0;
				x[17] = 1 << 5;
				x[18] = 1 << x[2];
				x[19] = 1 << ((x[ 3] ^                 x[16]) & 31);
				x[20] = 1;
				x[21] = 1 << ((                        x[18]) & 31);
				x[22] = 1 << ((                        x[19]) & 31);
				x[23] = 1 << ((                        x[20]) & 31);
				x[24] = 1 << ((                x[16] ^ x[21]) & 31);
				x[25] = 1 << ((                        x[22]) & 31);
				x[26] = 1 << ((                x[18] ^ x[23]) & 31);
				x[27] = 1 << ((                x[19] ^ x[24]) & 31);
				x[28] = 1 << ((                x[20] ^ x[25]) & 31);
				x[29] = 1 << ((                x[21] ^ x[26]) & 31);
				x[30] = 1 << ((        x[16] ^ x[22] ^ x[27]) & 31);
				x[31] = 1 << ((                x[23] ^ x[28]) & 31);
				EXPANSION(x,16); EXPANSION(x,17); EXPANSION(x,18); EXPANSION(x,19);
				EXPANSION(x,20); EXPANSION(x,21); EXPANSION(x,22); EXPANSION(x,23);
				EXPANSION(x,24); EXPANSION(x,25); EXPANSION(x,26); EXPANSION(x,27);
				EXPANSION(x,28); EXPANSION(x,29); EXPANSION(x,30); EXPANSION(x,31);
				EXPANSION(x,32); EXPANSION(x,33); EXPANSION(x,34); EXPANSION(x,35);
				EXPANSION(x,36); EXPANSION(x,37); EXPANSION(x,38); EXPANSION(x,39);
				EXPANSION(x,40); EXPANSION(x,41); EXPANSION(x,42); EXPANSION(x,43);
				EXPANSION(x,44); EXPANSION(x,45); EXPANSION(x,46); EXPANSION(x,47);
				EXPANSION(x,48); EXPANSION(x,49); EXPANSION(x,50); EXPANSION(x,51);
				EXPANSION(x,52); EXPANSION(x,53); EXPANSION(x,54); EXPANSION(x,55);
				EXPANSION(x,56); EXPANSION(x,57); EXPANSION(x,58); EXPANSION(x,59);
				EXPANSION(x,60); EXPANSION(x,61); EXPANSION(x,62); EXPANSION(x,63);

				a = hash0;
				b = hash1;
				c = hash2;
				d = hash3;
				e = hash4;

				SHA1_II_REV(b,c,d,e,a,x[79]); SHA1_II_REV(c,d,e,a,b,x[78]); SHA1_II_REV(d,e,a,b,c,x[77]); SHA1_II_REV(e,a,b,c,d,x[76]); SHA1_II_REV(a,b,c,d,e,x[75]);
				SHA1_II_REV(b,c,d,e,a,x[74]); SHA1_II_REV(c,d,e,a,b,x[73]); SHA1_II_REV(d,e,a,b,c,x[72]); SHA1_II_REV(e,a,b,c,d,x[71]); SHA1_II_REV(a,b,c,d,e,x[70]);
				SHA1_II_REV(b,c,d,e,a,x[69]); SHA1_II_REV(c,d,e,a,b,x[68]); SHA1_II_REV(d,e,a,b,c,x[67]); SHA1_II_REV(e,a,b,c,d,x[66]); SHA1_II_REV(a,b,c,d,e,x[65]);
				SHA1_II_REV(b,c,d,e,a,x[64]); SHA1_II_REV(c,d,e,a,b,x[63]); SHA1_II_REV(d,e,a,b,c,x[62]); SHA1_II_REV(e,a,b,c,d,x[61]); SHA1_II_REV(a,b,c,d,e,x[60]);

				SHA1_HH_REV(b,c,d,e,a,x[59]); SHA1_HH_REV(c,d,e,a,b,x[58]); SHA1_HH_REV(d,e,a,b,c,x[57]); SHA1_HH_REV(e,a,b,c,d,x[56]); SHA1_HH_REV(a,b,c,d,e,x[55]);
				SHA1_HH_REV(b,c,d,e,a,x[54]); SHA1_HH_REV(c,d,e,a,b,x[53]); SHA1_HH_REV(d,e,a,b,c,x[52]); SHA1_HH_REV(e,a,b,c,d,x[51]); SHA1_HH_REV(a,b,c,d,e,x[50]);
				SHA1_HH_REV(b,c,d,e,a,x[49]); SHA1_HH_REV(c,d,e,a,b,x[48]); SHA1_HH_REV(d,e,a,b,c,x[47]); SHA1_HH_REV(e,a,b,c,d,x[46]); SHA1_HH_REV(a,b,c,d,e,x[45]);
				SHA1_HH_REV(b,c,d,e,a,x[44]); SHA1_HH_REV(c,d,e,a,b,x[43]); SHA1_HH_REV(d,e,a,b,c,x[42]); SHA1_HH_REV(e,a,b,c,d,x[41]); SHA1_HH_REV(a,b,c,d,e,x[40]);

				SHA1_GG_REV(b,c,d,e,a,x[39]); SHA1_GG_REV(c,d,e,a,b,x[38]); SHA1_GG_REV(d,e,a,b,c,x[37]); SHA1_GG_REV(e,a,b,c,d,x[36]); SHA1_GG_REV(a,b,c,d,e,x[35]);
				SHA1_GG_REV(b,c,d,e,a,x[34]); SHA1_GG_REV(c,d,e,a,b,x[33]); SHA1_GG_REV(d,e,a,b,c,x[32]); SHA1_GG_REV(e,a,b,c,d,x[31]); SHA1_GG_REV(a,b,c,d,e,x[30]);
				SHA1_GG_REV(b,c,d,e,a,x[29]); SHA1_GG_REV(c,d,e,a,b,x[28]); SHA1_GG_REV(d,e,a,b,c,x[27]); SHA1_GG_REV(e,a,b,c,d,x[26]); SHA1_GG_REV(a,b,c,d,e,x[25]);
				SHA1_GG_REV(b,c,d,e,a,x[24]); SHA1_GG_REV(c,d,e,a,b,x[23]); SHA1_GG_REV(d,e,a,b,c,x[22]); SHA1_GG_REV(e,a,b,c,d,x[21]); SHA1_GG_REV(a,b,c,d,e,x[20]);

				SHA1_FF_REV(b,c,d,e,a,x[19]); SHA1_FF_REV(c,d,e,a,b,x[18]);

				SHA1_FF_REV(d,e,a,b,c,0); // 17
				t17a = a;
				t17b = b;
				t17c = c;
				t17d = d;
				t17e = e;

				if (x0 == 5)
				{
					break;
				}

				// x[17] = (1 << 5) to (1 << 31)
				for (; x[17] != 0; x[17] <<= 1, x[1] = ++x1 ^ x[3])
				{
					a = t17a;
					b = t17b;
					c = t17c - x[17];
					d = t17d;
					e = t17e;

					SHA1_FF_REV(e,a,b,c,d,x[16]);

					SHA1_FF_REV(a,b,c,d,e,    0); // 15
					SHA1_FF_REV(b,c,d,e,a,    0); SHA1_FF_REV(c,d,e,a,b,    0); SHA1_FF_REV(d,e,a,b,c,    0); SHA1_FF_REV(e,a,b,c,d,    0); SHA1_FF_REV(a,b,c,d,e,    0);
					SHA1_FF_REV(b,c,d,e,a,    0); SHA1_FF_REV(c,d,e,a,b,    0); SHA1_FF_REV(d,e,a,b,c,    0); SHA1_FF_REV(e,a,b,c,d,    0); SHA1_FF_REV(a,b,c,d,e,    0);
					SHA1_FF_REV(b,c,d,e,a,    0); // 4

					if (a == 0x59d148c0 && finishRev16Norm(b, c, d, e, x))
					{
						invalid = outputPassword(x + 4, onlyValid, onlyPrintable, hash, fout);
						invalidCount += invalid;
						count++;
						if (invalid == 0 && noCollisions)
						{
							return count - invalidCount;
						}
					}
				}
				x1 = 5;
				x[1] = 5 ^ x[3];
			}

			// x[17] = (1 << 5) to (1 << 31)
			for (; x[17] != 0; x[17] <<= 1, x[1] = ++x1 ^ x[3])
			{
				a = t17a;
				b = t17b;
				c = t17c - x[17];
				d = t17d;
				e = t17e;

				SHA1_FF_REV(e,a,b,c,d,0); // 16
				t16a = a;
				t16b = b;
				t16c = c;
				t16d = d;
				t16e = e;
				// x[16] = (1 << 5) to (1 << 31)
				for (; x[16] != 0; x[16] <<= 1, x[0] = ++x0 ^ x[2])
				{
					a = t16a;
					b = t16b;
					c = t16c;
					d = t16d - x[16];
					e = t16e;

					SHA1_FF_REV(a,b,c,d,e,    0); // 15
					SHA1_FF_REV(b,c,d,e,a,    0); SHA1_FF_REV(c,d,e,a,b,    0); SHA1_FF_REV(d,e,a,b,c,    0); SHA1_FF_REV(e,a,b,c,d,    0); SHA1_FF_REV(a,b,c,d,e,    0);
					SHA1_FF_REV(b,c,d,e,a,    0); SHA1_FF_REV(c,d,e,a,b,    0); SHA1_FF_REV(d,e,a,b,c,    0); SHA1_FF_REV(e,a,b,c,d,    0); SHA1_FF_REV(a,b,c,d,e,    0);
					SHA1_FF_REV(b,c,d,e,a,    0); // 4

					if (a == 0x59d148c0 && finishRev16Norm(b, c, d, e, x))
					{
						invalid = outputPassword(x + 4, onlyValid, onlyPrintable, hash, fout);
						invalidCount += invalid;
						count++;
						if (invalid == 0 && noCollisions)
						{
							return count - invalidCount;
						}
					}
				}
				x0 = 5;
				x[0] = 5 ^ x[2];
				x[16] = 1 << 5;
			}
			x1 = 5;
			x[1] = 5 ^ x[3];
		}
	}
	return count - invalidCount;
}

uint32_t finishRev16Norm(uint32_t t, uint32_t a, uint32_t b, uint32_t c, uint32_t *x)
{
	//x[4] = ROT(c, 32 - 30) - (0xc3d2e1f0 + ((0xefcdab89 & 0x98badcfe) | ((~0xefcdab89) & 0x10325476)) + ROT(0x67452301, 5) + 0x5a827999);
	x[4] = ROT(c, 32 - 30) - 0x9fb498b3;
	if ((x[4] & 31) == x[0])
	{
		b = ROT(b, 2);
		//x[5] = b - (0x10325476 + ((0x67452301 & 0x7bf36ae2) | ((~0x67452301) & 0x98badcfe)) + ROT(ROT(c, 32 - 30), 5) + 0x5a827999);
		x[5] = b - (ROT(ROT(c, 32 - 30), 5) + 0x66b0cd0d);
		if ((x[5] & 31) == x[1])
		{
			//x[6] = a - (0x98badcfe + ((ROT(c, 32 - 30) & 0x59d148c0) | ((~ROT(c, 32 - 30)) & 0x7bf36ae2)) + ROT(b, 5) + 0x5a827999);
			x[6] = a - (((ROT(c, 32 - 30) & 0x59d148c0) | ((~ROT(c, 32 - 30)) & 0x7bf36ae2)) + ROT(b, 5) + 0xf33d5697);
			if ((x[6] & 31) == x[2])
			{
				//x[7] = t - (0x7bf36ae2 + ((b & c) | ((~b) & 0x59d148c0)) + ROT(a, 5) + 0x5a827999);
				x[7] = t - (((b & c) | ((~b) & 0x59d148c0)) + ROT(a, 5) + 0xd675e47b);
				if ((x[7] & 31) == x[3])
				{
					x[8] = 0;
					x[9] = 0;
					return 1;
				}
			}
		}
	}
	return 0;
}

uint32_t XSHA1_rev20::calcNorm(const uint32_t *hash, uint32_t &invalidCount, bool onlyValid, bool onlyPrintable, bool noCollisions, FILE *fout)
{
	uint32_t a, b, c, d, e, x0, x1, x[80], count = 0, invalid;
	uint32_t hash0, hash1, hash2, hash3, hash4;
	uint32_t t17a, t17b, t17c, t17d, t17e;
	uint32_t t16a, t16b, t16c, t16d, t16e;

	invalidCount = 0;
	hash0 = hash[0] - 0x67452301;
	hash1 = hash[1] - 0xefcdab89;
	hash2 = hash[2] - 0x98badcfe;
	hash3 = hash[3] - 0x10325476;
	hash4 = hash[4] - 0xc3d2e1f0;

	for (x[4] = 0; x[4] < 32; x[4]++)
	{
		for (x[3] = 0; x[3] < 32; x[3]++)
		{
			for (x1 = 0; x1 < 5; x1++)
			{
				x[1] = x1 ^ x[3];
				for (x[2] = 0; x[2] < 32; x[2]++)
				{
					for (x0 = 0; /*x0 <= 5*/; x0++)
					{
						x[0] = x0 ^ x[2];
						x[16] = 1 << x0;
						x[17] = 1 << x1;
						x[18] = 1 << (x[2] ^ x[4]);
						x[19] = 1 << ((x[ 3] ^                 x[16]) & 31);
						x[20] = 1 << ((x[ 4] ^                 x[17]) & 31);
						x[21] = 1 << ((                        x[18]) & 31);
						x[22] = 1 << ((                        x[19]) & 31);
						x[23] = 1 << ((                        x[20]) & 31);
						x[24] = 1 << ((                x[16] ^ x[21]) & 31);
						x[25] = 1 << ((                x[17] ^ x[22]) & 31);
						x[26] = 1 << ((                x[18] ^ x[23]) & 31);
						x[27] = 1 << ((                x[19] ^ x[24]) & 31);
						x[28] = 1 << ((                x[20] ^ x[25]) & 31);
						x[29] = 1 << ((                x[21] ^ x[26]) & 31);
						x[30] = 1 << ((        x[16] ^ x[22] ^ x[27]) & 31);
						x[31] = 1 << ((        x[17] ^ x[23] ^ x[28]) & 31);
						EXPANSION(x,16); EXPANSION(x,17); EXPANSION(x,18); EXPANSION(x,19);
						EXPANSION(x,20); EXPANSION(x,21); EXPANSION(x,22); EXPANSION(x,23);
						EXPANSION(x,24); EXPANSION(x,25); EXPANSION(x,26); EXPANSION(x,27);
						EXPANSION(x,28); EXPANSION(x,29); EXPANSION(x,30); EXPANSION(x,31);
						EXPANSION(x,32); EXPANSION(x,33); EXPANSION(x,34); EXPANSION(x,35);
						EXPANSION(x,36); EXPANSION(x,37); EXPANSION(x,38); EXPANSION(x,39);
						EXPANSION(x,40); EXPANSION(x,41); EXPANSION(x,42); EXPANSION(x,43);
						EXPANSION(x,44); EXPANSION(x,45); EXPANSION(x,46); EXPANSION(x,47);
						EXPANSION(x,48); EXPANSION(x,49); EXPANSION(x,50); EXPANSION(x,51);
						EXPANSION(x,52); EXPANSION(x,53); EXPANSION(x,54); EXPANSION(x,55);
						EXPANSION(x,56); EXPANSION(x,57); EXPANSION(x,58); EXPANSION(x,59);
						EXPANSION(x,60); EXPANSION(x,61); EXPANSION(x,62); EXPANSION(x,63);

						a = hash0;
						b = hash1;
						c = hash2;
						d = hash3;
						e = hash4;

						SHA1_II_REV(b,c,d,e,a,x[79]); SHA1_II_REV(c,d,e,a,b,x[78]); SHA1_II_REV(d,e,a,b,c,x[77]); SHA1_II_REV(e,a,b,c,d,x[76]); SHA1_II_REV(a,b,c,d,e,x[75]);
						SHA1_II_REV(b,c,d,e,a,x[74]); SHA1_II_REV(c,d,e,a,b,x[73]); SHA1_II_REV(d,e,a,b,c,x[72]); SHA1_II_REV(e,a,b,c,d,x[71]); SHA1_II_REV(a,b,c,d,e,x[70]);
						SHA1_II_REV(b,c,d,e,a,x[69]); SHA1_II_REV(c,d,e,a,b,x[68]); SHA1_II_REV(d,e,a,b,c,x[67]); SHA1_II_REV(e,a,b,c,d,x[66]); SHA1_II_REV(a,b,c,d,e,x[65]);
						SHA1_II_REV(b,c,d,e,a,x[64]); SHA1_II_REV(c,d,e,a,b,x[63]); SHA1_II_REV(d,e,a,b,c,x[62]); SHA1_II_REV(e,a,b,c,d,x[61]); SHA1_II_REV(a,b,c,d,e,x[60]);

						SHA1_HH_REV(b,c,d,e,a,x[59]); SHA1_HH_REV(c,d,e,a,b,x[58]); SHA1_HH_REV(d,e,a,b,c,x[57]); SHA1_HH_REV(e,a,b,c,d,x[56]); SHA1_HH_REV(a,b,c,d,e,x[55]);
						SHA1_HH_REV(b,c,d,e,a,x[54]); SHA1_HH_REV(c,d,e,a,b,x[53]); SHA1_HH_REV(d,e,a,b,c,x[52]); SHA1_HH_REV(e,a,b,c,d,x[51]); SHA1_HH_REV(a,b,c,d,e,x[50]);
						SHA1_HH_REV(b,c,d,e,a,x[49]); SHA1_HH_REV(c,d,e,a,b,x[48]); SHA1_HH_REV(d,e,a,b,c,x[47]); SHA1_HH_REV(e,a,b,c,d,x[46]); SHA1_HH_REV(a,b,c,d,e,x[45]);
						SHA1_HH_REV(b,c,d,e,a,x[44]); SHA1_HH_REV(c,d,e,a,b,x[43]); SHA1_HH_REV(d,e,a,b,c,x[42]); SHA1_HH_REV(e,a,b,c,d,x[41]); SHA1_HH_REV(a,b,c,d,e,x[40]);

						SHA1_GG_REV(b,c,d,e,a,x[39]); SHA1_GG_REV(c,d,e,a,b,x[38]); SHA1_GG_REV(d,e,a,b,c,x[37]); SHA1_GG_REV(e,a,b,c,d,x[36]); SHA1_GG_REV(a,b,c,d,e,x[35]);
						SHA1_GG_REV(b,c,d,e,a,x[34]); SHA1_GG_REV(c,d,e,a,b,x[33]); SHA1_GG_REV(d,e,a,b,c,x[32]); SHA1_GG_REV(e,a,b,c,d,x[31]); SHA1_GG_REV(a,b,c,d,e,x[30]);
						SHA1_GG_REV(b,c,d,e,a,x[29]); SHA1_GG_REV(c,d,e,a,b,x[28]); SHA1_GG_REV(d,e,a,b,c,x[27]); SHA1_GG_REV(e,a,b,c,d,x[26]); SHA1_GG_REV(a,b,c,d,e,x[25]);
						SHA1_GG_REV(b,c,d,e,a,x[24]); SHA1_GG_REV(c,d,e,a,b,x[23]); SHA1_GG_REV(d,e,a,b,c,x[22]); SHA1_GG_REV(e,a,b,c,d,x[21]); SHA1_GG_REV(a,b,c,d,e,x[20]);

						SHA1_FF_REV(b,c,d,e,a,x[19]); SHA1_FF_REV(c,d,e,a,b,x[18]); SHA1_FF_REV(d,e,a,b,c,x[17]);

						if (x0 == 5)
						{
							break;
						}

						SHA1_FF_REV(e,a,b,c,d,x[16]);

						SHA1_FF_REV(a,b,c,d,e,    0); // 15
						SHA1_FF_REV(b,c,d,e,a,    0); SHA1_FF_REV(c,d,e,a,b,    0); SHA1_FF_REV(d,e,a,b,c,    0); SHA1_FF_REV(e,a,b,c,d,    0); SHA1_FF_REV(a,b,c,d,e,    0);
						SHA1_FF_REV(b,c,d,e,a,    0); SHA1_FF_REV(c,d,e,a,b,    0); SHA1_FF_REV(d,e,a,b,c,    0); SHA1_FF_REV(e,a,b,c,d,    0); SHA1_FF_REV(a,b,c,d,e,    0); // 5

						if (finishRev20Norm(a, b, c, d, e, x))
						{
							invalid = outputPassword(x + 5, onlyValid, onlyPrintable, hash, fout);
							invalidCount += invalid;
							count++;
							if (invalid == 0 && noCollisions)
							{
								return count - invalidCount;
							}
						}
					}

					SHA1_FF_REV(e,a,b,c,d,0); // 16
					t16a = a;
					t16b = b;
					t16c = c;
					t16d = d;
					t16e = e;
					// x[16] = (1 << 5) to (1 << 31)
					for (; x[16] != 0; x[16] <<= 1, x[0] = ++x0 ^ x[2])
					{
						a = t16a;
						b = t16b;
						c = t16c;
						d = t16d - x[16];
						e = t16e;

						SHA1_FF_REV(a,b,c,d,e,    0); // 15
						SHA1_FF_REV(b,c,d,e,a,    0); SHA1_FF_REV(c,d,e,a,b,    0); SHA1_FF_REV(d,e,a,b,c,    0); SHA1_FF_REV(e,a,b,c,d,    0); SHA1_FF_REV(a,b,c,d,e,    0);
						SHA1_FF_REV(b,c,d,e,a,    0); SHA1_FF_REV(c,d,e,a,b,    0); SHA1_FF_REV(d,e,a,b,c,    0); SHA1_FF_REV(e,a,b,c,d,    0); SHA1_FF_REV(a,b,c,d,e,    0); // 5

						if (finishRev20Norm(a, b, c, d, e, x))
						{
							invalid = outputPassword(x + 5, onlyValid, onlyPrintable, hash, fout);
							invalidCount += invalid;
							count++;
							if (invalid == 0 && noCollisions)
							{
								return count - invalidCount;
							}
						}
					}
				}
			}
			x[1] = 5 ^ x[3];
			for (x[2] = 0; x[2] < 32; x[2]++)
			{
				for (x0 = 0; /*x0 <= 5*/; x0++)
				{
					x[0] = x0 ^ x[2];
					x[16] = 1 << x0;
					x[17] = 1 << 5;
					x[18] = 1 << (x[2] ^ x[4]);
					x[19] = 1 << ((x[ 3] ^                 x[16]) & 31);
					x[20] = 1 << ((x[ 4]                        ) & 31);
					x[21] = 1 << ((                        x[18]) & 31);
					x[22] = 1 << ((                        x[19]) & 31);
					x[23] = 1 << ((                        x[20]) & 31);
					x[24] = 1 << ((                x[16] ^ x[21]) & 31);
					x[25] = 1 << ((                        x[22]) & 31);
					x[26] = 1 << ((                x[18] ^ x[23]) & 31);
					x[27] = 1 << ((                x[19] ^ x[24]) & 31);
					x[28] = 1 << ((                x[20] ^ x[25]) & 31);
					x[29] = 1 << ((                x[21] ^ x[26]) & 31);
					x[30] = 1 << ((        x[16] ^ x[22] ^ x[27]) & 31);
					x[31] = 1 << ((                x[23] ^ x[28]) & 31);
					EXPANSION(x,16); EXPANSION(x,17); EXPANSION(x,18); EXPANSION(x,19);
					EXPANSION(x,20); EXPANSION(x,21); EXPANSION(x,22); EXPANSION(x,23);
					EXPANSION(x,24); EXPANSION(x,25); EXPANSION(x,26); EXPANSION(x,27);
					EXPANSION(x,28); EXPANSION(x,29); EXPANSION(x,30); EXPANSION(x,31);
					EXPANSION(x,32); EXPANSION(x,33); EXPANSION(x,34); EXPANSION(x,35);
					EXPANSION(x,36); EXPANSION(x,37); EXPANSION(x,38); EXPANSION(x,39);
					EXPANSION(x,40); EXPANSION(x,41); EXPANSION(x,42); EXPANSION(x,43);
					EXPANSION(x,44); EXPANSION(x,45); EXPANSION(x,46); EXPANSION(x,47);
					EXPANSION(x,48); EXPANSION(x,49); EXPANSION(x,50); EXPANSION(x,51);
					EXPANSION(x,52); EXPANSION(x,53); EXPANSION(x,54); EXPANSION(x,55);
					EXPANSION(x,56); EXPANSION(x,57); EXPANSION(x,58); EXPANSION(x,59);
					EXPANSION(x,60); EXPANSION(x,61); EXPANSION(x,62); EXPANSION(x,63);

					a = hash0;
					b = hash1;
					c = hash2;
					d = hash3;
					e = hash4;

					SHA1_II_REV(b,c,d,e,a,x[79]); SHA1_II_REV(c,d,e,a,b,x[78]); SHA1_II_REV(d,e,a,b,c,x[77]); SHA1_II_REV(e,a,b,c,d,x[76]); SHA1_II_REV(a,b,c,d,e,x[75]);
					SHA1_II_REV(b,c,d,e,a,x[74]); SHA1_II_REV(c,d,e,a,b,x[73]); SHA1_II_REV(d,e,a,b,c,x[72]); SHA1_II_REV(e,a,b,c,d,x[71]); SHA1_II_REV(a,b,c,d,e,x[70]);
					SHA1_II_REV(b,c,d,e,a,x[69]); SHA1_II_REV(c,d,e,a,b,x[68]); SHA1_II_REV(d,e,a,b,c,x[67]); SHA1_II_REV(e,a,b,c,d,x[66]); SHA1_II_REV(a,b,c,d,e,x[65]);
					SHA1_II_REV(b,c,d,e,a,x[64]); SHA1_II_REV(c,d,e,a,b,x[63]); SHA1_II_REV(d,e,a,b,c,x[62]); SHA1_II_REV(e,a,b,c,d,x[61]); SHA1_II_REV(a,b,c,d,e,x[60]);

					SHA1_HH_REV(b,c,d,e,a,x[59]); SHA1_HH_REV(c,d,e,a,b,x[58]); SHA1_HH_REV(d,e,a,b,c,x[57]); SHA1_HH_REV(e,a,b,c,d,x[56]); SHA1_HH_REV(a,b,c,d,e,x[55]);
					SHA1_HH_REV(b,c,d,e,a,x[54]); SHA1_HH_REV(c,d,e,a,b,x[53]); SHA1_HH_REV(d,e,a,b,c,x[52]); SHA1_HH_REV(e,a,b,c,d,x[51]); SHA1_HH_REV(a,b,c,d,e,x[50]);
					SHA1_HH_REV(b,c,d,e,a,x[49]); SHA1_HH_REV(c,d,e,a,b,x[48]); SHA1_HH_REV(d,e,a,b,c,x[47]); SHA1_HH_REV(e,a,b,c,d,x[46]); SHA1_HH_REV(a,b,c,d,e,x[45]);
					SHA1_HH_REV(b,c,d,e,a,x[44]); SHA1_HH_REV(c,d,e,a,b,x[43]); SHA1_HH_REV(d,e,a,b,c,x[42]); SHA1_HH_REV(e,a,b,c,d,x[41]); SHA1_HH_REV(a,b,c,d,e,x[40]);

					SHA1_GG_REV(b,c,d,e,a,x[39]); SHA1_GG_REV(c,d,e,a,b,x[38]); SHA1_GG_REV(d,e,a,b,c,x[37]); SHA1_GG_REV(e,a,b,c,d,x[36]); SHA1_GG_REV(a,b,c,d,e,x[35]);
					SHA1_GG_REV(b,c,d,e,a,x[34]); SHA1_GG_REV(c,d,e,a,b,x[33]); SHA1_GG_REV(d,e,a,b,c,x[32]); SHA1_GG_REV(e,a,b,c,d,x[31]); SHA1_GG_REV(a,b,c,d,e,x[30]);
					SHA1_GG_REV(b,c,d,e,a,x[29]); SHA1_GG_REV(c,d,e,a,b,x[28]); SHA1_GG_REV(d,e,a,b,c,x[27]); SHA1_GG_REV(e,a,b,c,d,x[26]); SHA1_GG_REV(a,b,c,d,e,x[25]);
					SHA1_GG_REV(b,c,d,e,a,x[24]); SHA1_GG_REV(c,d,e,a,b,x[23]); SHA1_GG_REV(d,e,a,b,c,x[22]); SHA1_GG_REV(e,a,b,c,d,x[21]); SHA1_GG_REV(a,b,c,d,e,x[20]);

					SHA1_FF_REV(b,c,d,e,a,x[19]); SHA1_FF_REV(c,d,e,a,b,x[18]);

					SHA1_FF_REV(d,e,a,b,c,0); // 17
					t17a = a;
					t17b = b;
					t17c = c;
					t17d = d;
					t17e = e;

					if (x0 == 5)
					{
						break;
					}

					// x[17] = (1 << 5) to (1 << 31)
					for (; x[17] != 0; x[17] <<= 1, x[1] = ++x1 ^ x[3])
					{
						a = t17a;
						b = t17b;
						c = t17c - x[17];
						d = t17d;
						e = t17e;

						SHA1_FF_REV(e,a,b,c,d,x[16]);

						SHA1_FF_REV(a,b,c,d,e,    0); // 15
						SHA1_FF_REV(b,c,d,e,a,    0); SHA1_FF_REV(c,d,e,a,b,    0); SHA1_FF_REV(d,e,a,b,c,    0); SHA1_FF_REV(e,a,b,c,d,    0); SHA1_FF_REV(a,b,c,d,e,    0);
						SHA1_FF_REV(b,c,d,e,a,    0); SHA1_FF_REV(c,d,e,a,b,    0); SHA1_FF_REV(d,e,a,b,c,    0); SHA1_FF_REV(e,a,b,c,d,    0); SHA1_FF_REV(a,b,c,d,e,    0); // 5

						if (finishRev20Norm(a, b, c, d, e, x))
						{
							invalid = outputPassword(x + 5, onlyValid, onlyPrintable, hash, fout);
							invalidCount += invalid;
							count++;
							if (invalid == 0 && noCollisions)
							{
								return count - invalidCount;
							}
						}
					}
					x1 = 5;
					x[1] = 5 ^ x[3];
				}

				// x[17] = (1 << 5) to (1 << 31)
				for (; x[17] != 0; x[17] <<= 1, x[1] = ++x1 ^ x[3])
				{
					a = t17a;
					b = t17b;
					c = t17c - x[17];
					d = t17d;
					e = t17e;

					SHA1_FF_REV(e,a,b,c,d,0); // 16
					t16a = a;
					t16b = b;
					t16c = c;
					t16d = d;
					t16e = e;
					// x[16] = (1 << 5) to (1 << 31)
					for (; x[16] != 0; x[16] <<= 1, x[0] = ++x0 ^ x[2])
					{
						a = t16a;
						b = t16b;
						c = t16c;
						d = t16d - x[16];
						e = t16e;

						SHA1_FF_REV(a,b,c,d,e,    0); // 15
						SHA1_FF_REV(b,c,d,e,a,    0); SHA1_FF_REV(c,d,e,a,b,    0); SHA1_FF_REV(d,e,a,b,c,    0); SHA1_FF_REV(e,a,b,c,d,    0); SHA1_FF_REV(a,b,c,d,e,    0);
						SHA1_FF_REV(b,c,d,e,a,    0); SHA1_FF_REV(c,d,e,a,b,    0); SHA1_FF_REV(d,e,a,b,c,    0); SHA1_FF_REV(e,a,b,c,d,    0); SHA1_FF_REV(a,b,c,d,e,    0); // 5

						if (finishRev20Norm(a, b, c, d, e, x))
						{
							invalid = outputPassword(x + 5, onlyValid, onlyPrintable, hash, fout);
							invalidCount += invalid;
							count++;
							if (invalid == 0 && noCollisions)
							{
								return count - invalidCount;
							}
						}
					}
					x0 = 5;
					x[0] = 5 ^ x[2];
					x[16] = 1 << 5;
				}
				x1 = 5;
				x[1] = 5 ^ x[3];
			}
		}
	}
	return count - invalidCount;
}

uint32_t finishRev20Norm(uint32_t t, uint32_t a, uint32_t b, uint32_t c, uint32_t d, uint32_t *x)
{
	//x[5] = ROT(d, 32 - 30) - (0xc3d2e1f0 + ((0xefcdab89 & 0x98badcfe) | ((~0xefcdab89) & 0x10325476)) + ROT(0x67452301, 5) + 0x5a827999);
	x[5] = ROT(d, 32 - 30) - 0x9fb498b3;
	if ((x[5] & 31) == x[0])
	{
		//x[6] = ROT(c, 32 - 30) - (0x10325476 + ((0x67452301 & 0x7bf36ae2) | ((~0x67452301) & 0x98badcfe)) + ROT(ROT(d, 32 - 30), 5) + 0x5a827999);
		x[6] = ROT(c, 32 - 30) - (ROT(ROT(d, 32 - 30), 5) + 0x66b0cd0d);
		if ((x[6] & 31) == x[1])
		{
			b = ROT(b, 32 - 30);
			//x[7] = b - (0x98badcfe + ((ROT(d, 32 - 30) & 0x59d148c0) | ((~ROT(d, 32 - 30)) & 0x7bf36ae2)) + ROT(ROT(c, 32 - 30), 5) + 0x5a827999);
			x[7] = b - (((ROT(d, 32 - 30) & 0x59d148c0) | ((~ROT(d, 32 - 30)) & 0x7bf36ae2)) + ROT(ROT(c, 32 - 30), 5) + 0xf33d5697);
			if ((x[7] & 31) == x[2])
			{
				//x[8] = a - (0x7bf36ae2 + ((ROT(c, 32 - 30) & d) | ((~ROT(c, 32 - 30)) & 0x59d148c0)) + ROT(b, 5) + 0x5a827999);
				x[8] = a - (((ROT(c, 32 - 30) & d) | ((~ROT(c, 32 - 30)) & 0x59d148c0)) + ROT(b, 5) + 0xd675e47b);
				if ((x[8] & 31) == x[3])
				{
					//x[9] = t - (0x59d148c0 + ((b & c) | ((~b) & d)) + ROT(a, 5) + 0x5a827999);
					x[9] = t - (((b & c) | ((~b) & d)) + ROT(a, 5) + 0xb453c259);
					if ((x[9] & 31) == x[4])
					{
						x[10] = 0;
						return 1;
					}
				}
			}
		}
	}
	return 0;
}

uint32_t XSHA1_rev16::run(const uint32_t *hash, uint32_t &invalidCount, bool onlyValid, bool onlyPrintable, bool noCollisions, FILE *fout)
{
#ifdef ARC_x86
	if ((getInstructionSets() & (IS_AVX | IS_XOP)) == (IS_AVX | IS_XOP))
	{
		return calcAVXXOP(hash, invalidCount, onlyValid, onlyPrintable, noCollisions, fout);
	}
	else if ((getInstructionSets() & IS_AVX) == IS_AVX)
	{
		return calcAVX(hash, invalidCount, onlyValid, onlyPrintable, noCollisions, fout);
	}
	else if ((getInstructionSets() & IS_SSE2) == IS_SSE2)
	{
		return calcSSE2(hash, invalidCount, onlyValid, onlyPrintable, noCollisions, fout);
	}
#endif
	return calcNorm(hash, invalidCount, onlyValid, onlyPrintable, noCollisions, fout);
}

uint32_t XSHA1_rev20::run(const uint32_t *hash, uint32_t &invalidCount, bool onlyValid, bool onlyPrintable, bool noCollisions, FILE *fout)
{
#ifdef ARC_x86
	if ((getInstructionSets() & (IS_AVX | IS_XOP)) == (IS_AVX | IS_XOP))
	{
		return calcAVXXOP(hash, invalidCount, onlyValid, onlyPrintable, noCollisions, fout);
	}
	else if ((getInstructionSets() & IS_AVX) == IS_AVX)
	{
		return calcAVX(hash, invalidCount, onlyValid, onlyPrintable, noCollisions, fout);
	}
	else if ((getInstructionSets() & IS_SSE2) == IS_SSE2)
	{
		return calcSSE2(hash, invalidCount, onlyValid, onlyPrintable, noCollisions, fout);
	}
#endif
	return calcNorm(hash, invalidCount, onlyValid, onlyPrintable, noCollisions, fout);
}
