/*
	Copyright (C) 2012 Steve Thomas <steve AT tobtu DOT com>

	This file is part of XSHA1 Reverser.

	XSHA1 Reverser is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	XSHA1 Reverser is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with XSHA1 Reverser.  If not, see <http://www.gnu.org/licenses/>.
*/

// **** XSHA1 Reverser Version 1.1 ****
// This is called reversing because you are going backwards through the hashing algorithm.
// It also appears to "reverse" any data that is 16 bytes or less, but that's not exactly
// correct because of collisions.

#include <stdio.h>
#include <string.h>
#include <string>
#include <fstream>
#include "common.h"
#include "getargs.h"
#include "xsha1.h"

using namespace std;

void usage(const char *runStr);
int hex2bin(const char *hexStr, uint32_t *out, uint32_t length);
void xsha1(char *pw, uint32_t *hash);

int main(int argc, char *argv[])
{
	FILE *fout = stdout;
	programArgs args;
	int ret;
	uint32_t hash[5], count, invalidCount = 0;
	char pw[65];

	// Get arguments
	ret = validateArs(argc, argv, args);
	if (ret != 0 || args.help)
	{
		usage(argv[0]);
		return ret;
	}
	if (args.debugList)
	{
/*		printf(
			"MMX    = 0x0001 [not used] (64 bit integer instructions)\n"
			"SSE    = 0x0002 [not used]\n"
			"SSE2   = 0x0004            (128 bit integer instructions)\n"
			"SSE3   = 0x0008 [not used]\n"
			"SSSE3  = 0x0010 [not used] (pshufb)\n"
			"SSE41  = 0x0020 [not used] (ptest, using pmovmskb (SSE2) instead)\n"
			"SSE42  = 0x0040 [not used]\n"
			"SSE4A  = 0x0080 [not used]\n"
			"AVX    = 0x0100            (Better 128 bit integer instructions)\n"
			"AVX2   = 0x0200 [not used] (256 bit integer instructions :( Haswell in 2013)\n"
			"OS_YMM = 0x0400 [not used] (OS support for 256 bit AVX registers)\n"
			"XOP    = 0x0800            (vpcmov and vprotd)\n"
			"POPCNT = 0x1000 [not used] (popcnt)\n");*/
		printf(
			"These are the only flags checked (sorry no MMX or SSE4.1 [using pmovmskb\n"
			"instead of ptest]):\n"
			"SSE2   = 0x0004\n"
			"AVX    = 0x0100\n"
			"XOP    = 0x0800 (requires AVX so \"0x0900\")\n");
		return 0;
	}

	// Mask instruction sets with args.debugInt
	getInstructionSets(args.debugInt);

	// Open output file
	if (args.outputFile != NULL)
	{
		fout = fopen(args.outputFile, "w");
		if (fout == NULL)
		{
			perror("fopen output file");
			return 1;
		}
	}

	// Hash passwords
	for (uint32_t i = 0; i < args.xsha1.size(); i++)
	{
		if (strlen(args.xsha1[i]) > 64)
		{
			args.xsha1[i][64] = 0;
		}
		else
		{
			for (int j = 0; j < 16; j++)
			{
				((uint32_t*) pw)[j] = 0;
			}
		}
		strcpy(pw, args.xsha1[i]);
		xsha1(pw, hash);
		fprintf(fout, "%s:%08x%08x%08x%08x%08x\n", pw, hash[0], hash[1], hash[2], hash[3], hash[4]);
	}

	// Crack passwords
	if (args.hash.size() != 0 || args.hashList != NULL)
	{
		TIMER_TYPE t1, t2;

		TIMER_FUNC(t1);
		for (uint32_t i = 0; i < args.hash.size(); i++)
		{
			if (hex2bin(args.hash[i], hash, 5))
			{
				fprintf(stderr, "Error invalid hash: \"%s\"\n", args.hash[i]);
			}
			else
			{
				if (args.charactersInt == 20)
				{
					count = XSHA1_rev20::run(hash, invalidCount, !args.outputInvalid, args.onlyPrintable != 0, args.noCollisions != 0, fout);
				}
				else
				{
					count = XSHA1_rev16::run(hash, invalidCount, !args.outputInvalid, args.onlyPrintable != 0, args.noCollisions != 0, fout);
				}
				if (args.outputNotFound != 0 && count == 0)
				{
					fprintf(fout, "%08x%08x%08x%08x%08x:<not found>\n", hash[0], hash[1], hash[2], hash[3], hash[4]);
					if (count == 0 && args.hasInvalid != 0)
					{
						fprintf(fout, "Found %u invalid collision(s)\n", invalidCount);
					}
				}
			}
		}
		if (args.hashList != NULL)
		{
			string line;
			ifstream fin(args.hashList);
			uint32_t lines = 0, hashes = 0, nextHashes = args.statusInt;

			fin >> line;
			while (!fin.eof())
			{
				lines++;
				if (hex2bin(line.c_str(), hash, 5))
				{
					fprintf(stderr, "Error invalid hash line #%u: \"%s\"\n", lines, line.c_str());
				}
				else
				{
					if (args.charactersInt == 20)
					{
						count = XSHA1_rev20::run(hash, invalidCount, !args.outputInvalid, args.onlyPrintable != 0, args.noCollisions != 0, fout);
					}
					else
					{
						count = XSHA1_rev16::run(hash, invalidCount, !args.outputInvalid, args.onlyPrintable != 0, args.noCollisions != 0, fout);
					}
					if (args.outputNotFound != 0 && count == 0)
					{
						fprintf(fout, "%08x%08x%08x%08x%08x:<not found>\n", hash[0], hash[1], hash[2], hash[3], hash[4]);
						if (count == 0 && args.hasInvalid != 0)
						{
							fprintf(fout, "Found %u invalid collision(s)\n", invalidCount);
						}
					}
					hashes++;
					if (nextHashes == hashes)
					{
						printf("%u processed hashes\r", hashes);
						nextHashes += args.statusInt;
					}
				}
				fin >> line;
			}
		}
		TIMER_FUNC(t2);
		printf("Took %0.3f seconds\n", TIMER_DIFF(t1, t2));
	}
	return 0;
}

void usage(const char *runStr)
{
	printf("\n"
		"                         **** XSHA1 Reverser v1.1 ****\n\n"
		"Cracks XSHA1 hashes with speeds up to quattuordecillions of passwords per\n"
		"second (quattuordecillions is 10^45 \"quadrillion ^ 3\"). Any 16 or less\n"
		"character password takes about 15 ms and any 20 or less character password\n"
		"takes about 490 ms. Times will vary depending on hardware these times are from\n"
		"a 2.5 GHz Core2.\n\n"
		"%s [Options]\n\n"
		"Options:\n"
		"  -c|--characters=<int>\n"
		"    Number of characters to reverse [default is 16, can be 16 or 20].\n\n"
		"  -C|--no-collisions\n"
		"    Stops when first password is found.\n\n"
		"  --debug=<int>\n"
		"    Manually remove CPU instruction sets (this gets ANDed to the actual CPU\n"
		"    instruction sets).\n\n"
		"  --debug-list\n"
		"    List CPU instruction sets and exit.\n\n"
		"  -f|--output-not-found\n"
		"    Outputs <hash>:<not found> when hash is not cracked.\n\n"
		"  -h|--help\n"
		"    Displays help and exits.\n\n"
		"  -H|--hash=<string>\n"
		"    XSHA1 hash to be cracked. You can have multiple -H|--hash.\n\n"
		"  -i|--output-invalid\n"
		"    Outputs invalid passwords (invalid passwords contain A-Z and \\x00-\\x1f).\n\n"
		"  -I|--has-invalid\n"
		"    Outputs if it has invalid collisions if no valid password is found (ignored\n"
		"    when -i|--output-invalid is used or -f|--output-not-found is not used).\n\n"
		"  -l|--hash-list=<file>\n"
		"    A file with XSHA1 hashes in it.\n\n"
		"  -o|--output=<file>\n"
		"    Outputs to the file.\n\n"
		"  -p|--printable\n"
		"    Outputs only valid printable passwords (space through ~ except A-Z).\n\n"
		"  -s|--status=<int>\n"
		"    Display status every <int> hashes. This only applies to hashes from a hash\n"
		"    list file (-l|--hash-list) [default is to not display the status].\n\n"
		"  -x|--xsha1=<string>\n"
		"    Hashes the string with XSHA1. You can have multiple -x|--xsha1.\n\n"
		"Copyright (c) 2012 Steve Thomas <steve AT tobtu DOT com>\n"
		"  This is free software: you can redistribute it and/or modify it under the\n"
		"  terms of the GNU General Public License as published by the Free Software\n"
		"  Foundation, either version 3 of the License, or (at your option) any later\n"
		"  version. There is NO WARRANTY, to the extent permitted by law.\n", runStr);
}

int hex2bin(const char *hexStr, uint32_t *out, uint32_t length)
{
	uint32_t a, b, ch2;
	unsigned char ch;

	for (a = 0; hexStr[8 * a] != 0; a++)
	{
		out[a] = 0;
		for (b = 0; b < 8; b += 2)
		{
			ch = (unsigned char) hexStr[8 * a + b];
			if (ch >= '0' && ch <= '9')
			{
				ch2 = (ch - '0') << 4;
			}
			else if (ch >= 'A' && ch <= 'F')
			{
				ch2 = (ch - 'A' + 10) << 4;
			}
			else if (ch >= 'a' && ch <= 'f')
			{
				ch2 = (ch - 'a' + 10) << 4;
			}
			else
			{
				return 1;
			}
			ch = hexStr[8 * a + b + 1];
			if (ch >= '0' && ch <= '9')
			{
				ch2 |= ch - '0';
			}
			else if (ch >= 'A' && ch <= 'F')
			{
				ch2 |= ch - 'A' + 10;
			}
			else if (ch >= 'a' && ch <= 'f')
			{
				ch2 |= ch - 'a' + 10;
			}
			else
			{
				return 1;
			}
			out[a] = (out[a] << 8) | ch2;
		}
	}
	if (a != length)
	{
		return 1;
	}
	return 0;
}

#define ROT(n,s) ((n << (s)) | (n >> (32 - (s))))
void xsha1(char *pw, uint32_t *hash)
{
	uint32_t a = 0x67452301, b = 0xefcdab89, c = 0x98badcfe, d = 0x10325476, e = 0xc3d2e1f0, i, t, x[80] = {0};

	for (i = 0; i < 64; i++)
	{
		if (pw[i] >= 'A' && pw[i] <= 'Z')
		{
			pw[i] = pw[i] - 'A' + 'a';
		}
	}

	a = 0x67452301;
	b = 0xefcdab89;
	c = 0x98badcfe;
	d = 0x10325476;
	e = 0xc3d2e1f0;

	for (i = 0; i < 16; i++)
	{
		x[i] = ((uint32_t*)pw)[i];
	}
	for (i = 0; i < 64; i++)
	{
		x[i + 16] = 1 << ((x[i] ^ x[i + 2] ^ x[i + 8] ^ x[i + 13]) & 31);
	}
	for (i = 0; i < 80; i++)
	{
		if (i < 20)
		{
			t = e + ((b & c) | ((~b) & d))        + ROT(a, 5) + x[i] + 0x5a827999;
		}
		else if (i < 40)
		{
			t = e + (d ^ c ^ b)                   + ROT(a, 5) + x[i] + 0x6ed9eba1;
		}
		else if (i < 60)
		{
			t = e + ((c & b) | (d & c) | (d & b)) + ROT(a, 5) + x[i] + 0x8f1bbcdc;
		}
		else
		{
			t = e + (d ^ c ^ b)                   + ROT(a, 5) + x[i] + 0xca62c1d6;
		}

		e = d;
		d = c;
		c = ROT(b, 30);
		b = a;
		a = t;
	}

	hash[0] = a + 0x67452301;
	hash[1] = b + 0xefcdab89;
	hash[2] = c + 0x98badcfe;
	hash[3] = d + 0x10325476;
	hash[4] = e + 0xc3d2e1f0;
}
