/*
	Copyright (C) 2012 Steve Thomas <steve AT tobtu DOT com>

	This file is part of XSHA1 Reverser.

	XSHA1 Reverser is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	XSHA1 Reverser is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with XSHA1 Reverser.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string>
#include <vector>
#include <map>
#include "getargs.h"

using namespace std;

#ifdef _WIN32
	#include <string.h>
	#define USE_STR_TO_UL
	#define USE_STR_TO_ULL
	#define STR_TO_ULL(str,endPtr,radix) _strtoui64(str, endPtr, radix)
	#define STR_CMP_ICASE(a,b) strcmpi(a, b)
#else
	#include <strings.h>
	#define USE_STR_TO_UL
	#define USE_STR_TO_ULL
	#define STR_TO_ULL(str,endPtr,radix) strtoull(str, endPtr, radix)
	#define STR_CMP_ICASE(a,b) strcasecmp(a, b)
#endif

int searchArg(string &arg, char *value, map<string,char**> &argValues, map<string,vector<char*>*> &argValuesMulti, map<string,int*> &argSwitches);
int parseArgs(int argc, char *argv[], programArgs &args);
int validateInt(char *arg, uint32_t &out, bool required, const char *name, uint32_t rangeLo = 1, uint32_t rangeHi = 0, uint32_t defaultVal = 0, char endChar = 0, char **pos = NULL);

/**
 * return 0 all good
 * return 1 value used
 * return -1 error
 */
int searchArg(string &arg, char *value, map<string,char**> &argValues, map<string,vector<char*>*> &argValuesMulti, map<string,int*> &argSwitches)
{
	map<string,char**>::iterator itValue;
	map<string,vector<char*>*>::iterator itValueMulti;
	map<string,int*>::iterator itSwitch;
	string argName;
	size_t equal;
	int ret = 0;

	equal = arg.find('=');
	argName = arg.substr(0, equal);
	itValue = argValues.find(argName);
	if (itValue != argValues.end())
	{
		if (equal != string::npos)
		{
			*(itValue->second) = value;
		}
		else if (value != NULL)
		{
			*(itValue->second) = value;
			ret = 1;
		}
		else
		{
			fprintf(stderr, "Error: '-%s' requires a value\n", arg.c_str());
			ret = -1;
		}
	}
	else
	{
		itSwitch = argSwitches.find(argName);
		if (itSwitch != argSwitches.end())
		{
			if (equal == string::npos)
			{
				*(itSwitch->second) = 1;
			}
			else
			{
				fprintf(stderr, "Error: '-%s' does not have a value\n", argName.c_str());
				ret = -1;
			}
		}
		else
		{
			itValueMulti = argValuesMulti.find(argName);
			if (itValueMulti != argValuesMulti.end())
			{
				if (equal != string::npos)
				{
					itValueMulti->second->push_back(value);
					ret = 0;
				}
				else if (value != NULL)
				{
					itValueMulti->second->push_back(value);
					ret = 1;
				}
				else
				{
					fprintf(stderr, "Error: '-%s' requires a value\n", arg.c_str());
					ret = -1;
				}
			}
			else
			{
				fprintf(stderr, "Error: Unknown argument '-%s'\n", argName.c_str());
				ret = -1;
			}
		}
	}
	return ret;
}

int parseArgs(int argc, char *argv[], programArgs &args)
{
	map<string,char**> argValues;
	map<string,vector<char*>*> argValuesMulti;
	map<string,int*> argSwitches;
	string arg;
	char *value;
	size_t equal;
	int a, b, ret = 0, tmp;

	args.noCollisions   = 0;
	args.debugList      = 0;
	args.outputNotFound = 0;
	args.help           = 0;
	args.outputInvalid  = 0;
	args.hasInvalid     = 0;
	args.onlyPrintable  = 0;

	args.characters     = NULL;
	args.debug          = NULL;
	args.hashList       = NULL;
	args.outputFile     = NULL;
	args.status         = NULL;
	args.hash.clear();
	args.xsha1.clear();

	argSwitches["-no-collisions"]    = &(args.noCollisions);
	argSwitches["C"]                 = &(args.noCollisions);
	argSwitches["-debug-list"]       = &(args.debugList);
	argSwitches["-output-not-found"] = &(args.outputNotFound);
	argSwitches["f"]                 = &(args.outputNotFound);
	argSwitches["-help"]             = &(args.help);
	argSwitches["h"]                 = &(args.help);
	argSwitches["-output-invalid"]   = &(args.outputInvalid);
	argSwitches["i"]                 = &(args.outputInvalid);
	argSwitches["-has-invalid"]      = &(args.hasInvalid);
	argSwitches["I"]                 = &(args.hasInvalid);
	argSwitches["-printable"]        = &(args.onlyPrintable);
	argSwitches["p"]                 = &(args.onlyPrintable);
	argValues["-characters"]         = &(args.characters);
	argValues["c"]                   = &(args.characters);
	argValues["-debug"]              = &(args.debug);
	argValues["-hash-list"]          = &(args.hashList);
	argValues["l"]                   = &(args.hashList);
	argValues["-output"]             = &(args.outputFile);
	argValues["o"]                   = &(args.outputFile);
	argValues["-status"]             = &(args.status);
	argValues["s"]                   = &(args.status);
	argValuesMulti["-hash"]          = &(args.hash);
	argValuesMulti["H"]              = &(args.hash);
	argValuesMulti["-xsha1"]         = &(args.xsha1);
	argValuesMulti["x"]              = &(args.xsha1);

	for (a = 1; a < argc; a++)
	{
		if (argv[a][0] == '-')
		{
			value = NULL;
			if (argv[a][1] == '-')
			{
				arg = argv[a] + 1;
				equal = arg.find('=');
				if (equal != string::npos)
				{
					value = argv[a] + (equal + 2);
				}
				else if (a + 1 < argc)
				{
					value = argv[a + 1];
				}
				switch (searchArg(arg, value, argValues, argValuesMulti, argSwitches))
				{
					case  1: a++; break;
					case -1: ret = 1;
				}
			}
			else
			{
				for (b = 1; argv[a][b] != 0 && argv[a][b] != '='; b++)
				{
					arg = argv[a][b];
					if (argv[a][b + 1] == '=')
					{
						arg = argv[a] + b;
						value = argv[a] + (b + 2);
					}
					else if (argv[a][b + 1] == 0 && a + 1 < argc)
					{
						value = argv[a + 1];
					}
					tmp = searchArg(arg, value, argValues, argValuesMulti, argSwitches);
					if (tmp == 1)
					{
						a++;
						break;
					}
					else if (tmp != 0)
					{
						ret = 1;
						break;
					}
				}
			}
		}
		else
		{
			fprintf(stderr, "Error: Unknown argument '%s'\n", argv[a]);
			ret = 1;
		}
	}
	return ret;
}

int validateInt(char *arg, uint32_t &out, bool required, const char *name, uint32_t rangeLo, uint32_t rangeHi, uint32_t defaultVal, char endChar, char **pos)
{
#ifdef USE_STR_TO_UL
	char *endPtr;
#endif
	uint32_t val = 0;
	int ret = 0;

	if (arg == NULL)
	{
		out = defaultVal;
		if (required)
		{
			fprintf(stderr, "Error: Missing argument %s\n", name);
			ret = 1;
		}
	}
	else
	{
#ifdef USE_STR_TO_UL
		errno = 0;
		out = val = (uint32_t) strtoul(arg, &endPtr, 0);
		if ((*endPtr != endChar && *endPtr != 0) || errno != 0)
		{
			ret = 1;
		}
		if (pos != NULL)
		{
			*pos = endPtr;
		}
#else
		if (*arg == '0')
		{
			if (arg[1] == 'x')
			{
				for (arg += 2; *arg != endChar || *arg != 0; arg++)
				{
					if (*arg >= '0' && *arg <= '9')
					{
						val = 16 * val + (uint32_t) (*arg - '0');
					}
					else if (*arg >= 'a' && *arg <= 'f')
					{
						val = 16 * val + (uint32_t) (*arg - 'a') + 10;
					}
					else if (*arg >= 'A' && *arg <= 'F')
					{
						val = 16 * val + (uint32_t) (*arg - 'A') + 10;
					}
					else
					{
						ret = 1;
						break;
					}
				}
			}
			else
			{
				for (arg++; *arg != endChar || *arg != 0; arg++)
				{
					if (*arg >= '0' && *arg <= '7')
					{
						val = 8 * val + (uint32_t) (*arg - '0');
					}
					else
					{
						ret = 1;
						break;
					}
				}
			}
		}
		else
		{
			for (; *arg != endChar || *arg != 0; arg++)
			{
				if (*arg >= '0' && *arg <= '9')
				{
					val = 10 * val + (uint32_t) (*arg - '0');
				}
				else
				{
					ret = 1;
					break;
				}
			}
		}
		if (pos != NULL)
		{
			*pos = arg;
		}
#endif
		if (rangeLo > rangeHi)
		{
			if (ret)
			{
				fprintf(stderr, "Error: Invalid argument %s must be an integer\n", name);
			}
		}
		else if (ret || val < rangeLo || val > rangeHi)
		{
			fprintf(stderr, "Error: Invalid argument %s must be an integer between %u and %u\n", name, rangeLo, rangeHi);
			ret = 1;
		}
	}
	return ret;
}

int validateArs(int argc, char *argv[], programArgs &args)
{
	int ret;

	if (argc == 1)
	{
		return 1;
	}
	ret = parseArgs(argc, argv, args);
	if (args.help || args.debugList)
	{
		return ret;
	}

	if (args.hashList == NULL && args.hash.size() == 0 && args.xsha1.size() == 0)
	{
		fprintf(stderr, "Error: Missing one or more arguments -H|--hash, -l|--hash-list, and/or -x|--xsha1\n");
		ret = 1;
	}
	if (args.outputInvalid && args.onlyPrintable)
	{
		fprintf(stderr, "Error: Can't output invalid passwords and only output printable passwords (-i|--output-invalid and -p|--printable)\n");
		ret = 1;
	}
	ret |= validateInt(args.characters, args.charactersInt, false, "-c|--characters", 1,      0, 16);
	ret |= validateInt(args.debug,      args.debugInt,      false, "--debug",         0, IS_ALL, 0xffffffff);
	ret |= validateInt(args.status,     args.statusInt,     false, "-s|--status");
	return ret;
}
