/*
	Copyright (C) 2012 Steve Thomas <steve AT tobtu DOT com>

	This file is part of XSHA1 Reverser.

	XSHA1 Reverser is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	XSHA1 Reverser is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with XSHA1 Reverser.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <stdio.h>
#include "common.h"

uint32_t outputPassword(uint32_t *in, bool onlyValid, bool onlyPrintable, const uint32_t *hash, FILE *fout)
{
	int len = 20;
	uint32_t last = in[4];
	char ch;

	if (in[4] == 0)
	{
		len -= 4;
		last = in[3];
		if (in[3] == 0)
		{
			len -= 4;
			last = in[2];
			if (in[2] == 0)
			{
				len -= 4;
				last = in[1];
				if (in[1] == 0)
				{
					len -= 4;
					last = in[0];
				}
			}
		}
	}
	if (last == 0)
	{
		fprintf(fout, "%08x%08x%08x%08x%08x::\n", hash[0], hash[1], hash[2], hash[3], hash[4]);
		return 0;
	}
	if ((last & 0xff000000) == 0)
	{
		len--;
		if ((last & 0x00ff0000) == 0)
		{
			len--;
			if ((last & 0x0000ff00) == 0)
			{
				len--;
			}
		}
	}

	// valid check
	if (onlyPrintable)
	{
		for (int a = 0; a < len; a++)
		{
			if (((char*)in)[a] > '~' || ((char*)in)[a] < ' ' || (((char*)in)[a] >= 'A' && ((char*)in)[a] <= 'Z'))
			{
				return 1;
			}
		}
	}
	else if (onlyValid)
	{
		for (int a = 0; a < len; a++)
		{
			if (((char*)in)[a] < ' ' || (((char*)in)[a] >= 'A' && ((char*)in)[a] <= 'Z'))
			{
				return 1;
			}
		}
	}

	fprintf(fout, "%08x%08x%08x%08x%08x:", hash[0], hash[1], hash[2], hash[3], hash[4]);
	for (int a = 0; a < len; a++)
	{
		ch = ((((char*)in)[a] >> 4) & 0xf) + '0';
		if (ch > '9')
		{
			ch += 'a' - '9' - 1;
		}
		fprintf(fout, "%c", ch);
		ch = (((char*)in)[a] & 0xf) + '0';
		if (ch > '9')
		{
			ch += 'a' - '9' - 1;
		}
		fprintf(fout, "%c", ch);
		if (((char*)in)[a] < ' ')
		{
			((char*)in)[a] = '.';
		}
	}
	fprintf(fout, ":%s\n", (char*) in);
	return 0;
}

uint32_t getInstructionSets(uint32_t debugSet)
{
#ifdef ARC_x86
	static uint32_t ret = 0xffffffff;

	if (ret == 0xffffffff)
	{
		uint32_t out1, out2, out3, out4, out5;

#ifdef _WIN32
		__asm
		{
			mov   eax,0x80000001
			cpuid
			mov   eax,1
			mov   out3,ecx
			mov   out4,edx
			cpuid
			mov   out1,ecx
			mov   out2,edx
		}
#else
		asm(
			"movl   $2147483649,%%eax\n\t"
			"cpuid\n\t"
			"movl   $1,%%eax\n\t"
			"movl   %%ecx,%2\n\t"
			"movl   %%edx,%3\n\t"
			"cpuid"
			: "=c"(out1), "=d"(out2), "=m"(out3), "=m"(out4) // output
			: // input
			: "eax", "ebx"); // used
#endif
		ret = 0;
		if (out2 & (1 << 23))
		{
			ret |= IS_MMX;
		}
		if (out2 & (1 << 25))
		{
			ret |= IS_SSE;
		}
		if (out2 & (1 << 26))
		{
			ret |= IS_SSE2;
		}
		if (out1 & (1 << 0))
		{
			ret |= IS_SSE3;
		}
		if (out1 & (1 << 9))
		{
			ret |= IS_SSSE3;
		}
		if (out1 & (1 << 19))
		{
			ret |= IS_SSE41;
		}
		if (out1 & (1 << 20))
		{
			ret |= IS_SSE42;
		}
		if (out3 & (1 << 6))
		{
			ret |= IS_SSE4A;
		}
		if (out1 & (1 << 28))
		{
			ret |= IS_AVX;
		}
		if ((ret & IS_AVX) && (out1 & (1 << 27))) // OSXSAVE (XGETBV)
		{
#ifdef _WIN32
			__asm
			{
				xor   ecx,ecx
				xgetbv
				mov   out5,eax
			}
#else
			asm(
				"xor   %%ecx,%%ecx\n\t"
				"xgetbv"
				: "=a"(out5) // output
				: // input
				: "ecx", "edx"); // used
#endif
			if ((out5 & 0x6) == 0x6)
			{
				ret |= IS_OS_YMM;
			}
		}
		if (ret & IS_OS_YMM)
		{
#ifdef _WIN32
			__asm
			{
				mov   eax,7
				xor   ecx,ecx
				cpuid
				mov   out5,ebx
			}
#else
			asm(
				"movl   $7,%%eax\n\t"
				"xor   %%ecx,%%ecx\n\t"
				"cpuid\n\t"
				: "=b"(out5) // output
				: // input
				: "eax", "ecx", "edx"); // used
#endif
			if (out5 & (1 << 5))
			{
				ret |= IS_AVX2;
			}
		}
		if (out3 & (1 << 11))
		{
			ret |= IS_XOP;
		}
		if (out1 & (1 << 23))
		{
			ret |= IS_POPCNT;
		}
#ifdef ARC_x86_32
		if ((out2 & (1 << 30)) || (out4 & (1 << 29)) || (out3 & (1 << 6))) // x86_64, long_mode, long_mode_intel
		{
			fprintf(stderr, "WARNING: **** Your CPU can run 64 bit which runs faster than the 32 bit code you are currently running. ****\n");
		}
#endif
	}
	if (debugSet != 0xffffffff)
	{
		ret &= debugSet;
	}
	return ret;
#else
	return 0;
#endif
}
