﻿Imports System
Imports System.Collections.Generic
Imports System.Text
Imports System.IO
Imports System.Drawing
Imports System.Drawing.Imaging
Imports System.Diagnostics
Imports System.Globalization
Public NotInheritable Class BniIcon
  Implements IDisposable
  Private m_img As Image
  Private m_flags As Integer
  Private m_softwareList As String()
  Friend Sub New(img As Image, flags As Integer, softwareList As UInteger())
    m_img = img
    m_flags = flags
    m_softwareList = New String(softwareList.Length - 1) {}
    For I As Integer = 0 To softwareList.Length - 1
      Dim code As Byte() = BitConverter.GetBytes(softwareList(I))
      Dim temp As Byte = code(0)
      code(0) = code(3)
      code(3) = temp
      temp = code(1)
      code(1) = code(2)
      code(2) = temp
      m_softwareList(I) = Encoding.ASCII.GetString(code)
    Next
  End Sub
  Public ReadOnly Property Image() As Image
    Get
      Return m_img
    End Get
  End Property
  Public ReadOnly Property UserFlags() As Integer
    Get
      Return m_flags
    End Get
  End Property
  Public ReadOnly Property SoftwareProductCodes() As String()
    Get
      Dim codes As String() = New String(m_softwareList.Length - 1) {}
      Array.Copy(m_softwareList, codes, codes.Length)
      Return codes
    End Get
  End Property
  Public Overrides Function ToString() As String
    Dim sb As New StringBuilder()
    If m_softwareList.Length > 0 Then
      sb.Append(m_softwareList(0))
      For I As Integer = 1 To m_softwareList.Length - 1
        sb.AppendFormat(CultureInfo.CurrentCulture, ",{0}", m_softwareList)
      Next
    End If
    Return String.Format(CultureInfo.CurrentCulture, "Flags: 0x{0:x8}, Products: {1}", m_flags, sb)
  End Function
#Region "IDisposable Members"
  Public Sub Dispose() Implements System.IDisposable.Dispose
    Dispose(True)
    GC.SuppressFinalize(Me)
  End Sub
  Private Sub Dispose(disposing As Boolean)
    If disposing Then
      If m_img IsNot Nothing Then
        m_img.Dispose()
        m_img = Nothing
      End If
    End If
  End Sub
#End Region
End Class
Public NotInheritable Class BniFileParser
  Implements IDisposable
#Region "utility types"
  Private Structure IconHeader
    Public flagValue As UInteger, width As UInteger, height As UInteger
    Public software As UInteger()
    Public Sub New(br As BinaryReader)
      flagValue = br.ReadUInt32()
      width = br.ReadUInt32()
      height = br.ReadUInt32()
      Dim soft As New List(Of UInteger)()
      Dim val As UInteger
      Do
        val = br.ReadUInt32()
        If val <> 0 Then soft.Add(val)
      Loop While val <> 0
      software = soft.ToArray()
    End Sub
  End Structure
  Private Enum StartDescriptor As Byte
    BottomLeft = 0
    BottomRight = &H10
    TopLeft = &H20
    TopRight = &H30
  End Enum
#End Region
  Private m_fullImage As Image
  Private m_icons As New List(Of BniIcon)()
  Public Sub New(filePath As String)
    If Not File.Exists(filePath) Then Throw New FileNotFoundException("The specified file does not exist.", filePath)
    Using fs As New FileStream(filePath, FileMode.Open, FileAccess.Read, FileShare.Read)
      Parse(fs)
    End Using
  End Sub
  Public Sub New(bniFileStream As Stream)
    If Object.ReferenceEquals(Nothing, bniFileStream) Then Throw New ArgumentNullException("bniFileStream")
    Parse(bniFileStream)
  End Sub
  Private Sub Parse(str As Stream)
    Using br As New BinaryReader(str)
      br.ReadUInt32()
      Dim version As UShort = br.ReadUInt16()
      If version <> 1 Then Throw New InvalidDataException("Only version 1 of BNI files is supported.")
      br.ReadUInt16()
      Dim numIcons As UInteger = br.ReadUInt32()
      br.ReadUInt32()
      Dim icons As New List(Of IconHeader)(CInt(numIcons))
      For I As Integer = 0 To numIcons - 1
        icons.Add(New IconHeader(br))
      Next
      Dim infoLength As Byte = br.ReadByte()
      br.ReadByte()
      If br.ReadByte() <> &HA Then Throw New InvalidDataException("Only run-length true-color TGA icons are supported.")
      br.ReadBytes(5)
      br.ReadUInt16()
      br.ReadUInt16()
      Dim width As UShort = br.ReadUInt16()
      Dim height As UShort = br.ReadUInt16()
      Dim depth As Byte = br.ReadByte()
      If depth <> 24 Then Throw New InvalidDataException("Only 24-bit TGA is supported.")
      Dim descriptor As StartDescriptor = DirectCast(br.ReadByte(), StartDescriptor)
      Dim info_bytes As Byte() = br.ReadBytes(infoLength)
      Dim numberOfPixels As Integer = width * height
      Using bmp As New Bitmap(width, height)
        Dim data As BitmapData = bmp.LockBits(New Rectangle(Point.Empty, New Size(width, height)), ImageLockMode.[WriteOnly], PixelFormat.Format32bppArgb)
        Dim currentPixel As IntPtr = data.Scan0
        Dim pI As Integer = 0
        While pI < numberOfPixels
          Dim packetType As Byte = br.ReadByte()
          Dim packetPixelCount As Byte = CByte((packetType And &H7F) + 1)
          If (packetType And &H80) = &H80 Then
            Dim nextBlue As Byte = br.ReadByte()
            Dim nextGreen As Byte = br.ReadByte()
            Dim nextRed As Byte = br.ReadByte()
            Dim c As Color = Color.FromArgb(255, nextRed, nextGreen, nextBlue)
            For pixel As Integer = 0 To packetPixelCount - 1
              System.Runtime.InteropServices.Marshal.WriteInt32(currentPixel, c.ToArgb)
              currentPixel = IntPtr.Add(currentPixel, 4)
            Next
          Else
            For pixel As Integer = 0 To packetPixelCount - 1
              Dim nextBlue As Byte = br.ReadByte()
              Dim nextGreen As Byte = br.ReadByte()
              Dim nextRed As Byte = br.ReadByte()
              Dim c As Color = Color.FromArgb(255, nextRed, nextGreen, nextBlue)
              System.Runtime.InteropServices.Marshal.WriteInt32(currentPixel, c.ToArgb)
              currentPixel = IntPtr.Add(currentPixel, 4)
            Next
          End If
          pI += packetPixelCount
        End While
        currentPixel = Nothing
        bmp.UnlockBits(data)
        m_fullImage = New Bitmap(width, height)
        If descriptor = StartDescriptor.TopRight Then
          bmp.RotateFlip(RotateFlipType.RotateNoneFlipX)
        ElseIf descriptor = StartDescriptor.BottomLeft Then
          bmp.RotateFlip(RotateFlipType.RotateNoneFlipY)
        ElseIf descriptor = StartDescriptor.BottomRight Then
          bmp.RotateFlip(RotateFlipType.RotateNoneFlipXY)
        End If
        Using g As Graphics = Graphics.FromImage(m_fullImage)
          g.DrawImage(bmp, Point.Empty)
        End Using
        Dim currentY As Integer = 0
        For I As Integer = 0 To numIcons - 1
          Dim header As IconHeader = icons(I)
          Dim icon As New Bitmap(bmp, CInt(width), CInt(header.height))
          Using g As Graphics = Graphics.FromImage(icon)
            Dim nextIcon As New Size(CInt(width), CInt(header.height))
            g.DrawImage(bmp, New Rectangle(Point.Empty, nextIcon), New Rectangle(New Point(0, currentY), nextIcon), GraphicsUnit.Pixel)
          End Using
          Dim curIcon As New BniIcon(icon, CInt(header.flagValue), header.software)
          m_icons.Add(curIcon)
          currentY += CInt(header.height)
        Next
      End Using
    End Using
  End Sub
  Public ReadOnly Property FullImage() As Image
    Get
      Return m_fullImage
    End Get
  End Property
  Public ReadOnly Property AllIcons() As BniIcon()
    Get
      Return m_icons.ToArray()
    End Get
  End Property
#Region "IDisposable Members"
  Public Sub Dispose() Implements System.IDisposable.Dispose
    Dispose(True)
    GC.SuppressFinalize(Me)
  End Sub
  Private Sub Dispose(disposing As Boolean)
    If disposing Then
      If m_fullImage IsNot Nothing Then
        m_fullImage.Dispose()
        m_fullImage = Nothing
      End If
    End If
  End Sub
#End Region
End Class




